/*
 * Copyright (c) 1993  Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 */

#include <stdio.h>
#include <string.h>
#include <lber.h>
#include <ldap.h>
#include "ud.h"

char *bound_dn;		/* bound user's Distinguished Name */

extern LDAPMessage * find();

#ifdef DEBUG
extern int debug;
#endif

extern int verbose;
extern struct entry2 Entry;
extern LDAP *ld;
extern char bound_as[];
extern char *group_base;

void add_group(name)
char *name;
{
	register char **vp;
	register int i, idx = 0;
	char tmp[BUFSIZ], dn[BUFSIZ];
	static LDAPMod *attrs[9];
	LDAPMod init_rdn,    init_owner,   init_domain,
		init_errors, init_request, init_joinable;
	char *init_rdn_value[2], *init_owner_value[2], *init_domain_value[2],
	  	*init_errors_value[MAX_VALUES], *init_joinable_value[2],
		*init_request_value[MAX_VALUES];
	extern char ** fetch_attrs();
	extern void ldap_flush_cache();
	extern char * strip_ignore_chars();

#ifdef DEBUG
	if (debug & D_TRACE)
		printf("->add_group(%s)\n", name);
#endif

	if (!strcmp(bound_as, DEFAULT_BOUND_AS)) {
		printf("  You must first 'bind' in order to create a group.\n\n");
		if (auth((char *) NULL, (char *) NULL) < 0) {
			return;
		}
	}

	/*
	 *  If the user did not supply us with a name, prompt them for
	 *  a name.
	 */
	if ((name == NULL) || (*name == '\0')) {
		printf("  Group to create? ");
		fflush(stdout);
		fetch_buffer(tmp, sizeof(tmp), stdin);
		if (tmp[0] == '\0') {
			return;
		}
		name = strdup(tmp);
	}

	/* remove quotes, dots, and underscores. */
	name = strip_ignore_chars(name);
	sprintf(dn, "cn=%s, %s", name, group_base);

	/*
	 *  Make sure that this group does not already exist.
	 */
	if (vrfy(dn) == TRUE) {
		printf("  The group \"%s\" already exists.\n", name);
		return;
	}

	/*
	 *  Take the easy way out:  Fill in some reasonable values for
	 *  the most important fields, and make the user use the modify
	 *  command to change them, or to give values to other fields.
	 */
	init_rdn_value[0] = name;
	init_rdn_value[1] = NULL;
	init_rdn.mod_op = LDAP_MOD_ADD;
	init_rdn.mod_type = "cn";
	init_rdn.mod_values = init_rdn_value;
	attrs[idx++] = &init_rdn;

	init_owner_value[0] = bound_dn;
	init_owner_value[1] = NULL;
	init_owner.mod_op = LDAP_MOD_ADD;
	init_owner.mod_type = "owner";
	init_owner.mod_values = init_owner_value;
	attrs[idx++] = &init_owner;

#ifdef UOFM
	init_domain_value[0] = "umich.edu";
#else
	init_domain_value[0] = ".";
#endif
	init_domain_value[1] = NULL;
	init_domain.mod_op = LDAP_MOD_ADD;
	init_domain.mod_type = "associatedDomain";
	init_domain.mod_values = init_domain_value;
	attrs[idx++] = &init_domain;

	if ((vp = fetch_attrs(bound_dn, "mail")) == NULL) {
		sprintf(tmp, "postmaster@%s", init_domain_value[0]);
		init_errors_value[0] = strdup(tmp);
		i = 1;
	}
	else {
		register char **rcpp = vp;
		for (i = 0; (*rcpp != NULL) && (i < MAX_VALUES); i++)
			init_errors_value[i] = *rcpp++;
	}
	if (i != 0) {
		init_errors_value[i] = NULL;
		init_errors.mod_op = LDAP_MOD_ADD;
		init_errors.mod_type = "rfc822ErrorsTo";
		init_errors.mod_values = init_errors_value;
		attrs[idx++] = &init_errors;
	}

	for (i = 0; (init_errors_value[i] != NULL) && (i < MAX_VALUES); i++)
		init_request_value[i] = init_errors_value[i];
	if (i != 0) {
		init_request_value[i] = NULL;
		init_request.mod_op = LDAP_MOD_ADD;
		init_request.mod_type = "rfc822RequestsTo";
		init_request.mod_values = init_request_value;
		attrs[idx++] = &init_request;
	}

	init_joinable_value[0] = "FALSE";
	init_joinable_value[1] = NULL;
	init_joinable.mod_op = LDAP_MOD_ADD;
	init_joinable.mod_type = "joinable";
	init_joinable.mod_values = init_joinable_value;
	attrs[idx++] = &init_joinable;

	/* end it with a NULL */
	attrs[idx] = NULL;

#ifdef DEBUG
	if (debug & D_GROUPS) {
		register LDAPMod **lpp;
		register char **cpp;
		register int j;
		extern char * code_to_str();
		printf("  About to call ldap_add()\n");
		printf("  ld = 0x%x\n", ld);
		printf("  dn = [%s]\n", dn);
		for (lpp = attrs, i = 0; *lpp != NULL; lpp++, i++) {
			printf("  attrs[%1d]  code = %s  type = %s\n", i, 
				code_to_str((*lpp)->mod_op), (*lpp)->mod_type);
			for (cpp = (*lpp)->mod_values, j = 0; *cpp != NULL; cpp++, j++)
				printf("    value #%1d = %s\n", j, *cpp);
			printf("    value #%1d = NULL\n", j);
		}
	}
#endif

	/*
	 *  Now add this to the X.500 Directory.
	 */
	if (ldap_add_s(ld, dn, attrs) != 0) {
		ldap_perror(ld, "  ldap_add_s");
		printf("  Group not added.\n");
		return;
	}
	if (verbose)
		printf("  Group \"%s\" has been added to the Directory\n",
		       name);

	/*
	 *  We need to blow away the cache here.
	 *
	 *  Since we first looked up the name before trying to create it,
	 *  and that look-up failed, the cache will falsely claim that this
	 *  entry does not exist.
	 */
	(void) ldap_flush_cache(ld);
	if (vp != (char **) NULL)
		(void) ldap_value_free(vp);
	return;
}

void remove_group(name)
char *name;
{
	char dn[BUFSIZ], tmp[BUFSIZ];
	extern char * strip_ignore_chars();

#ifdef DEBUG
	if (debug & D_TRACE)
		printf("->remove_group(%s)\n", name);
#endif
	if (!strcmp(bound_as, DEFAULT_BOUND_AS)) {
		printf("  You must first 'bind' in order to remove a group.\n\n");
		if (auth((char *) NULL, (char *) NULL) < 0)
			return;
	}
	/*
	 *  If the user did not supply us with a name, prompt them for
	 *  a name.
	 */
	if ((name == NULL) || (*name == '\0')) {
		printf("Group to remove? ");
		fflush(stdout);
		fetch_buffer(tmp, sizeof(tmp), stdin);
		if (tmp[0] == '\0')
			return;
		name = tmp;
	}
	name = strip_ignore_chars(name);
	sprintf(dn, "cn=%s, %s", name, group_base);

	/*
	 *  Now remove this from the X.500 Directory.
	 */
	if (ldap_delete_s(ld, dn) != 0) {
		ldap_perror(ld, "  ldap_add_s");
		printf("  Group not removed.\n");
		return;
	}
	ldap_uncache_entry(ld, dn);
	if (verbose)
		printf("  The group \"%s\" has been removed.\n", name);
	return;
}

void x_group(action, name)
int action;
char *name;
{
	char **vp, tmp[BUFSIZ], dn[BUFSIZ];
	char *values[2];
	LDAPMod mod, *mods[2];
	LDAPMessage *lm;
	static char *actions[] = { "join", "resign from", NULL };

#ifdef DEBUG
	if (debug & D_TRACE)
		printf("->x_group(%d, %s)\n", action, name);
#endif

	switch (action) {
	case G_JOIN:
	case G_RESIGN:
		break;
	default:
		printf("x_group:  %d is not a known action\n", action);
	}

	if (!strcmp(bound_as, DEFAULT_BOUND_AS)) {
		printf("  You must first 'bind' in order to %s a group.\n", actions[action]);
		if (auth((char *) NULL, (char *) NULL) < 0)
			return;
	}

	/*
	 *  If the user did not supply us with a name, prompt them for
	 *  a name.
	 */
	if ((name == NULL) || (*name == '\0')) {
		printf("  Group to %s? ", actions[action]);
		fflush(stdout);
		fetch_buffer(tmp, sizeof(tmp), stdin);
		if (tmp[0] == '\0') {
			return;
		}
		name = tmp;
	}
	sprintf(dn, "cn=%s, %s", name, group_base);

#ifdef DEBUG
	if (debug & D_GROUPS)
		printf("Group DN = (%s)\n", dn);
#endif

	/* make sure the group in question exists and is joinable */
	if ((lm = find(name)) == (LDAPMessage *) NULL) {
		printf("  Could not locate group \"%s\"\n", name);
		return;
	}
	parse_answer(lm);
	vp = Entry.attrs[attr_to_index("joinable")].values;
	if (action == G_JOIN) {
		if (vp == NULL) {
			printf("  No one is permitted to join \"%s\"\n", name);
			return;
		}
		if (!strcasecmp(*vp, "FALSE")) {
			printf("  No one is permitted to join \"%s\"\n", name);
			return;
		}
	}

	/*  fill in the ldap_mod() structure */
	mods[0] = &mod;
	mods[1] = (LDAPMod *) NULL;

	values[0] = Entry.DN;
	values[1] = NULL;
	switch (action) {
	case G_JOIN:
		mod.mod_op = LDAP_MOD_ADD;
		break;
	case G_RESIGN:
		mod.mod_op = LDAP_MOD_DELETE;
		break;
	}
	mod.mod_type = "memberOfGroup";
	mod.mod_values = values;

#ifdef DEBUG
	if (debug & D_GROUPS) {
		register LDAPMod **lpp;
		register char **cp;
		register int i, j;
		printf("  About to call ldap_modify_s()\n");
		printf("  ld = 0x%x\n", ld);
		printf("  dn = [%s]\n", bound_dn);
		for (lpp = mods, i = 1; *lpp != NULL; lpp++, i++) {
			printf("  mods[%1d] code = %1d\n", i, (*lpp)->mod_op);
			printf("  mods[%1d] type = %s\n", i, (*lpp)->mod_type);
			for (cp = (*lpp)->mod_values, j = 1; *cp != NULL; cp++, j++)
				printf("  mods[%1d] v[%1d] = %s\n", i, j, *cp);
				
		}
	}
#endif

	if (ldap_modify_s(ld, bound_dn, mods)) {
		if ((action == G_JOIN) && (ld->ld_errno == LDAP_TYPE_OR_VALUE_EXISTS))
			printf("  You are already a member of \"%s\"\n", name);
		else if ((action == G_RESIGN) && (ld->ld_errno == LDAP_NO_SUCH_ATTRIBUTE))
			printf("  You are not a member of \"%s\"\n", name);
		else
			ldap_perror(ld, "  ldap_modify_s");
		return;
	}
	ldap_uncache_entry(ld, bound_dn);
	if (verbose) {
		switch (action) {
		case G_JOIN:
			printf("  You are now a member of \"%s\"\n", name);
			break;
		case G_RESIGN:
			printf("  You are no longer a member of \"%s\"\n", name);
			break;
		}
	}
	return;
}

void do_members(group)
char *group;
{
	char s[BUFSIZ], *member, *values[2];
	LDAPMod mod, *mods[2];
	LDAPMessage *mp;

#ifdef DEBUG
	if (debug & D_TRACE)
		printf("->do_members(%s)\n", group);
#endif
	/*
	 *  At this point the user can indicate that he wishes to add members
	 *  to the group, delete members from the group, or replace the 
	 *  current list of members with a new list.  The first two cases
	 *  are very straight-forward, but the last case requires a little
	 *  extra care and work.
	 */
	if (verbose) {
		printf("\n");
		format("There are three options available at this point.  You may:  Add additional members to your group; Delete members from your group; or Replace the entire list of members with a new list.\n", 75, 2);
	}
	for (;;) {
		printf("  Do you want to Add members, Delete members, or Replace the list? ");
		fflush(stdout);
		fetch_buffer(s, sizeof(s), stdin);
		if (s[0] == '\0') {
			return;
			/*NOTREACHED*/
		}
		if (!strncasecmp(s, "add", strlen(s))) {
			mod.mod_op = LDAP_MOD_ADD;
			break;
		}
		else if (!strncasecmp(s, "delete", strlen(s))) {
			mod.mod_op = LDAP_MOD_DELETE;
			break;
		}
		else if (!strncasecmp(s, "replace", strlen(s))) {
			mod.mod_op = LDAP_MOD_REPLACE;
			break;
		}
		else if (verbose) {
			printf("\n");
			format("Did not recognize that response.  Please use 'A' to add members, 'D' to delete members, or 'R' to replace the entire list with a new list\n", 75, 2);
		}
	}
	if (mod.mod_op == LDAP_MOD_REPLACE)
		printf("  Member list will be overwritten.  Please specify a new member list.\n");
	if (verbose) {
		printf("\n");
		format("Members may be specified as a name (which is then looked up in the X.500 Directory) or as a domain-style (i.e., user@domain) e-mail address.  Simply hit the RETURN key at the \"Member?\" prompt when finished.\n", 75, 2);
		printf("\n");
	}

	for (;;) {
		printf("Member? ");
		fflush(stdout);
		fetch_buffer(s, sizeof(s), stdin);
		if (s[0] == '\0')
			return;

		/*
	 	 *  If the string the user has just typed has an @-sign in it,
	 	 *  then we assume it is an e-mail address.  In this case, we
	 	 *  just store away whatever it is they have typed.
	 	 *
	 	 *  If the string had no @-sign, then we look in the Directory,
	 	 *  make sure it exists, and if it does, we add that.
	 	 */
		if (strchr(s, '@') == NULL) {
			if ((mp = find(s)) == (LDAPMessage *) NULL) {
				printf("  Could not find \"%s\"\n", s);
				continue;
			}
			parse_answer(mp);
			member = Entry.DN;
			mod.mod_type = "member";
		}
		else {
			member = s;
			mod.mod_type = "mail";
		}

		/*  fill in the rest of the ldap_mod() structure */
		mods[0] = &mod;
		mods[1] = (LDAPMod *) NULL;

		values[0] = member;
		values[1] = NULL;
		mod.mod_values = values;

#ifdef DEBUG
		if (debug & D_GROUPS) {
			register LDAPMod **lpp;
			register char **cp;
			register int i, j;
			printf("  About to call ldap_modify_s()\n");
			printf("  ld = 0x%x\n", ld);
			printf("  dn = [%s]\n", group);
			for (lpp = mods, i = 1; *lpp != NULL; lpp++, i++) {
				printf("  mods[%1d] code = %1d\n", 
							i, (*lpp)->mod_op);
				printf("  mods[%1d] type = %s\n", 
							i, (*lpp)->mod_type);
				for (cp = (*lpp)->mod_values, j = 1; *cp != NULL; cp++, j++)
					printf("  mods[%1d] v[%1d] = %s\n", 
								i, j, *cp);
			}
		}
#endif

		if (ldap_modify_s(ld, group, mods)) {
			if (ld->ld_errno == LDAP_NO_SUCH_ATTRIBUTE) {
				printf("  Could not locate member \"%s\"\n", 
								member);
				continue;
			}
			else {
				ldap_perror(ld, "ldap_modify_s");
				return;
			}
		}
		ldap_uncache_entry(ld, group);

		/*
	 	 *  If the operation was REPLACE, we now need to "zero out" the
	 	 *  other "half" of the list (e.g., user specified an e-mail
	 	 *  address; now we need to clear the DN part of the list).
		 *
		 *  NOTE:  WE HAVE ALREADY DONE HALF OF THE REPLACE ABOVE.
	 	 *
	 	 *  Also, change the opcode to LDAP_MOD_ADD and give the user an
	 	 *  opportunity to add additional members to the group.  We
	 	 *  only take this branch the very first time during a REPLACE
	 	 *  operation.
	 	 */
		if (mod.mod_op == LDAP_MOD_REPLACE) {
			if (!strcmp(mod.mod_type, "member"))
				mod.mod_type = "mail";
			else
				mod.mod_type = "member";
			mods[0] = &mod;
			values[0] = NULL;
			mod.mod_values = values;
			mod.mod_op = LDAP_MOD_DELETE;
#ifdef DEBUG
			if (debug & D_GROUPS) {
				register LDAPMod **lpp;
				register char **cp;
				register int i, j;
				printf("  About to call ldap_modify_s()\n");
				printf("  ld = 0x%x\n", ld);
				printf("  dn = [%s]\n", group);
				for (lpp = mods, i = 1; *lpp != NULL; lpp++, i++) {
					printf("  mods[%1d] code = %1d\n", 
							i, (*lpp)->mod_op);
					printf("  mods[%1d] type = %s\n", 
							i, (*lpp)->mod_type);
					for (cp = (*lpp)->mod_values, j = 1; *cp != NULL; cp++, j++)
						printf("  mods[%1d] v[%1d] = %s\n", i, j, *cp);
				}
			}
#endif
			if (ldap_modify_s(ld, group, mods)) {
				/*
			 	*  A "No such attribute" error is no big deal.
			 	*  We only wanted to clear the attribute anyhow.
			 	*/
				if (ld->ld_errno != LDAP_NO_SUCH_ATTRIBUTE) {
					ldap_perror(ld, "ldap_modify_s");
					return;
				}
			}
			ldap_uncache_entry(ld, group);
			if (verbose)
				printf("  \"%s\" is now a member\n", member);
			mod.mod_op = LDAP_MOD_ADD;
		}
		else if (verbose && (mod.mod_op == LDAP_MOD_ADD))
			printf("  \"%s\" is now a member\n", member);
		else if (verbose && (mod.mod_op == LDAP_MOD_DELETE))
			printf("  \"%s\" is no longer a member\n", member);
	}
}

void purge_group(group)
char *group;
{
	int isclean = TRUE;
	LDAPMessage *lm;
	LDAPMod mod, *mods[2];
	char dn[BUFSIZ], tmp[BUFSIZ], *values[2], **vp;

#ifdef DEBUG
	if (debug & D_TRACE)
		printf("->purge_group(%s)\n", group);
#endif
	if (!strcmp(bound_as, DEFAULT_BOUND_AS)) {
		printf("  You must first 'bind' in order to purge a group.\n\n");
		if (auth((char *) NULL, (char *) NULL) < 0)
			return;
	}
	/*
	 *  If the user did not supply us with a name, prompt them for
	 *  a name.
	 */
	if ((group == NULL) || (*group == '\0')) {
		printf("Group to purge? ");
		fflush(stdout);
		fetch_buffer(tmp, sizeof(tmp), stdin);
		if (tmp[0] == '\0')
			return;
		group = tmp;
	}
	sprintf(dn, "cn=%s, %s", group, group_base);

	/* make sure the group in question exists */
	if ((lm = find(group)) == (LDAPMessage *) NULL) {
		printf("  Could not locate group \"%s\"\n", group);
		return;
	}
	parse_answer(lm);

	/* none of this stuff changes */
	mods[0] = &mod;
	mods[1] = (LDAPMod *) NULL;

	values[1] = NULL;

	mod.mod_values = values;
	mod.mod_type = "member";
	mod.mod_op = LDAP_MOD_DELETE;

	/*
	 *  Now cycle through all of the names in the "members" part of the
	 *  group (but not the e-mail address part).  Lookup each one, and
	 *  if it isn't found, let the user know so s/he can delete it.
	 */
	vp = Entry.attrs[attr_to_index("member")].values;
	for (; *vp != NULL; vp++) {
		char ans[BUFSIZ], *ufn, *label = "Did not find:  ";
		int len = strlen(label);

		if (vrfy(*vp))
			continue;
		isclean = FALSE;
		ufn = ldap_dn2ufn(*vp);
		format2(ufn, label, (char *) NULL, 2, 2 + len, 80);
		free(ufn);
ask:
		printf("  Purge, Keep, Abort [Keep]? ");
		fflush(stdout);
		fetch_buffer(ans, sizeof(ans), stdin);
		if ((ans[0] == '\0') || !strncasecmp(ans, "Keep", strlen(ans)))
			continue;
		if (!strncasecmp(ans, "Abort", strlen(ans))) {
			ldap_uncache_entry(ld, dn);
			return;
		}
		if (!strncasecmp(ans, "Purge", strlen(ans))) {
			values[0] = *vp;
#ifdef DEBUG
			if (debug & D_GROUPS) {
				register LDAPMod **lpp;
				register char **cp;
				register int i, j;
				printf("  About to call ldap_modify_s()\n");
				printf("  ld = 0x%x\n", ld);
				printf("  dn = [%s]\n", Entry.DN);
				for (lpp = mods, i = 1; *lpp != NULL; lpp++, i++) {
					printf("  mods[%1d] code = %1d\n", i, (*lpp)->mod_op);
					printf("  mods[%1d] type = %s\n", i, (*lpp)->mod_type);
					for (cp = (*lpp)->mod_values, j = 1; *cp != NULL; cp++, j++)
						printf("  mods[%1d] v[%1d] = %s\n", i, j, *cp);
						
				}
			}
#endif
			if (ldap_modify_s(ld, Entry.DN, mods))
				ldap_perror(ld, "  ldap_modify_s");
		}
		else {
			printf("  Did not recognize that answer.\n\n");
			goto ask;
		}
	}
	ldap_uncache_entry(ld, dn);
	if (isclean)
		printf("  No entries were purged.\n");
	return;
}

void tidy_up()
{
	register int i = 0;
	int found_one;
	register char **vp;
	LDAPMessage *lm, *gm;
	static LDAPMod mod;
	static LDAPMod *mods[2] = { &mod };
	static char *values[MAX_VALUES];

#ifdef DEBUG
	if (debug & D_TRACE)
		printf("->tidy()\n");
#endif

	if (!strcmp(bound_as, DEFAULT_BOUND_AS)) {
		printf("  You must first 'bind' in order to tidy your entry.\n\n");
		if (auth((char *) NULL, (char *) NULL) < 0) {
			return;
		}
	}

	/* lookup the user, and see to which groups he has subscribed */
	if ((lm = find(bound_as)) == (LDAPMessage *) NULL) {
		printf("  Could not locate \"%s\"\n", bound_as);
		return;
	}
	parse_answer(lm);
	vp = Entry.attrs[attr_to_index("memberOfGroup")].values;
	if (vp == NULL) {
		printf("  You have not subscribed to any groups.\n");
		return;
	}

	/* now, loop through these groups, deleting the bogus */
	for ( ; *vp != NULL; vp++) {
		if (vrfy(*vp))
			continue;
		found_one++;
		printf("  \"%s\" is not a valid group name.\n", *vp);
		values[i++] = strdup(*vp);
	}
	if (found_one == 0)
		return;

	/* delete the most heinous entries */
	values[i] = NULL;
	mod.mod_values = values;
	mod.mod_op = LDAP_MOD_DELETE;
	mod.mod_type = "memberOfGroup";
	if (ldap_modify_s(ld, bound_dn, mods))
		ldap_perror(ld, "ldap_modify_s");
	ldap_uncache_entry(ld, bound_dn);

	/* tidy up before we finish tidy_up */
	for ( ; i >= 1; i--)
		(void) free(values[i - 1]);
	return;
}
