/*
 * Copyright (c) 1991, 1992, 1993 
 * Regents of the University of Michigan.  All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#ifndef __STDC__
#include <memory.h>
#endif
#include <lber.h>
#include <ldap.h>
#include "ud.h"

extern char *search_base;	/* search base */
extern int verbose;		/* verbose mode flag */
static int num_picked = 0;	/* used when user picks entry at More prompt */
static LDAPMessage *res;	/* results returned from ldap_search_s() */
extern LDAP *ld;		/* our ldap descriptor */

#ifdef DEBUG
extern int debug;		/* debug flag */
#endif

vrfy(dn)
char *dn;
{
	static char *attrs[2] = { "objectClass", NULL };

#ifdef DEBUG
	if (debug & D_TRACE)
		printf("->vrfy(%s)\n", dn);
#endif
	/* verify that this DN exists in the directory */
	(void) ldap_search_s(ld, dn, LDAP_SCOPE_BASE, "objectClass=*", attrs, TRUE, &res);
	(void) ldap_msgfree(res);
	if ((ld->ld_errno == LDAP_NO_SUCH_OBJECT) || (ld->ld_errno == LDAP_INVALID_DN_SYNTAX))
		return(0);
	else if (ld->ld_errno == LDAP_SUCCESS)
		return(1);
	else {
		ldap_perror(ld, "ldap_search");
		return(0);
	}
}
	

LDAPMessage * find(who)
char *who;
{
	register char *cp;		/* general char pointer */
	register char *ap;		/* temp pointer for algorithm */
	register int i;			/* general int */
	int sidx = 0;			/* # of search algorithm */
	int in_gap = FALSE;		/* for parsing up the name */
	int matches;			/* from ldap_count_entries() */
	int choice;			/* entry that user chooses */
	int tried_fuzzy;		/* tried a fuzzy match */
	static char response[RESP_SIZE];	/* results from user */
	static char *namelist[MAX_NUM_NAMES];	/* names found */
	static char filter[BUFSIZ];		/* search filter */
	static char algorithm[BUFSIZ];		/* search algorithm */
	static char name[BUFSIZ];		/* DN to lookup */
	static int rc;			/* return from ldap_search */
	register char *rcp;		/* for stripping quotes, */

	struct parsed_name {
		int fields;
		char *comp[MAX_NAME_COMPS];
		char *name;			/* chopped up bits */
		char *entire;			/* complete name */
	} WHO;

	/* read attributes */
	static char *read_attrs[] = { "cn", "title", "postalAddress", 
		"telephoneNumber", "mail", "homePhone", "homePostalAddress", 
		"objectClass", "joinable", "member", "description", "owner", 
		"associatedDomain", "rfc822RequestsTo", "rfc822ErrorsTo",
#ifdef UOFM
		"multiLineDescription",
#endif
#ifdef KERBEROS
		"krbName",
#endif
		"facsimileTelephoneNumber", "uid", "memberOfGroup", NULL };

	/* search attributes */
	static char *attrs[] = { "cn", "title", "objectClass", "mail", NULL };

	/* search algorithms */
	extern char *search_algorithm[];

	extern char * strip_ignore_chars();

#ifdef DEBUG
	if (debug & D_TRACE)
		fprintf(stderr, "->find(%s)\n", who);
#endif

	/*
	 *  If the user-supplied name has any commas in it, we
	 *  assume that it is a UFN, and do everything right
	 *  here.
	 */
	if (strchr(who, ',') != NULL) {
#ifdef DEBUG
		if (debug & D_FIND)
			printf("\"%s\" appears to be a UFN\n", who);
#endif
		if ((rc = ldap_ufn_search_s(ld, who, attrs, FALSE, &res)) !=
		    LDAP_SUCCESS && rc != LDAP_SIZELIMIT_EXCEEDED &&
		    rc != LDAP_TIMELIMIT_EXCEEDED) {
			ldap_perror(ld, "ldap_ufn_search_s");
			return(NULL);
		}
		if ((matches = ldap_count_entries(ld, res)) < 0) {
			ldap_perror(ld, "ldap_count_entries");
			return(NULL);
		}
		if (matches != 1)
			return(NULL);
		if (ldap_search_s(ld, ldap_get_dn(ld, ldap_first_entry(ld, res)), LDAP_SCOPE_BASE, "objectClass=*", read_attrs, FALSE, &res) != LDAP_SUCCESS) {
			ldap_perror(ld, "ldap_search_s");
			return(NULL);
		}
		return(res);
	}

	/*
	 *  This program was modelled after a dreadful program which runs
	 *  on the mainframe here at U-M which requires the user to either
	 *  quote the name, or to separate the parts by dots or underbars.
	 *  We should just strip them because, while we are dreadful, we
	 *  aren't *that* dreadful.
	 */
	who = strip_ignore_chars(who);
			
	/*
	 *  Parse up the name awk-style, and fill in the structure
	 *  we use to record this.  Skip over any leading blank characters
	 *  while we are here too.
	 */
	for (rcp = who; isnamesepartor(*rcp); rcp++)
		;
	i = 0;
	WHO.name = strdup(rcp);
	WHO.entire = strdup(rcp);
	WHO.comp[i++] = WHO.name;
	WHO.fields = 1;
	for (rcp = WHO.name; *rcp != '\0'; ) {

		if (isnamesepartor(*rcp) && !in_gap) {
			in_gap = TRUE;
			*rcp++ = '\0';
		}
		else if (!isnamesepartor(*rcp) && in_gap) {
			/* new component */
			in_gap = FALSE;
			WHO.comp[i++] = rcp;
			WHO.fields++;
			rcp++;
		}
		else {
			rcp++;
		}
	}
#ifdef DEBUG
	if (debug & D_FIND) {
		printf("  Entire = [%s]\n", WHO.entire);
		printf("  Fields = %d  ", WHO.fields);
		for (i = 0; i < WHO.fields; i++)
			printf("[%s] ", WHO.comp[i]);
		printf("\n");
	}
#endif

try_again:
	/* if we have exhausted our search algorithms, give up */
	if (search_algorithm[sidx] == NULL)
		return(NULL);
	tried_fuzzy = FALSE;

	/*
	 *  See if the next search algorithm applies to this name.
	 *  If not, skip it, and move on to the next one.  If it
	 *  does apply, set the search filter to the apporpriate value.
	 */
	strcpy(algorithm, search_algorithm[sidx]);
	ap = algorithm;
#ifdef DEBUG
	if (debug & D_FIND)
		printf("Using algorithm \"%s\"\n", ap);
#endif
	if (algorithm[0] == '[') {
		if (algorithm[2] != ']') {
			printf("  Unparsable algorithm \"%s\"\n",
				search_algorithm[sidx]);
			sidx++;
			goto try_again;
		}
		if ((algorithm[1] - '0') != WHO.fields) {
			sidx++;
			goto try_again;
		}
		ap = &algorithm[3];
	}
	while (isspace(*ap))
		ap++;
#ifdef DEBUG
	if (debug & D_FIND)
		printf("  Raw filter = \"%s\"\n", ap);
#endif
	for (cp = filter; *ap != '\0'; ap++) {
		switch (*ap) {
		case '$' :
			ap++;
			if (isdigit(*ap))
				i = *ap - '0';
			else if (!strncasecmp(ap, "NF", 2)) {
				i = WHO.fields;
				ap++;
			}
			else {
				printf("  Unparsable algorithm \"%s\"\n",
					search_algorithm[sidx]);
				sidx++;
				goto try_again;
			}
			if (i > WHO.fields) {
				/*
				 *  This algorithm makes reference to a name
				 *  component ($N) which is greater than the
				 *  number of fields actually in the name.
				 */
				sidx++;
				goto try_again;
			}
			/*
			 *  Don't screw around with the index is it is the
			 *  "magic number" of 0.
			 */
			if (i > 0)
				i--;
			if (i == 0) {
				for (rcp = WHO.entire; *rcp != '\0'; rcp++)
					*cp++ = *rcp;
			}
			else {
				for (rcp = WHO.comp[i]; *rcp != '\0'; rcp++)
					*cp++ = *rcp;
			}
			break;
		case '~' :
			tried_fuzzy = TRUE; /* and fall through */
		default :
			*cp++ = *ap;
		}
	}
	*cp = '\0';
#ifdef DEBUG
	if (debug & D_FIND)
		printf("  Cooked filter \"%s\"\n", filter);
#endif
		
	/*
	 *  Now perform the search and count the results.
	 */
#ifdef DEBUG
	if (debug & D_FIND) {
		printf("  Calling ldap_search_s()\n");
		printf("     ld = 0x%x\n", ld);
		printf("     search base = %s\n", search_base);
		printf("     scope = LDAP_SCOPE_SUBTREE\n");
		printf("     filter = %s\n", filter);
		for (i = 0; attrs[i] != NULL; i++)
			printf("     attrs[%d] = %s\n", i, attrs[i]);
		printf("     attrs[%d] = NULL\n", i);
		printf("     attrsonly = FALSE\n");
		printf("     &results = 0x%x\n", &res);
	}
#endif
	if ((rc = ldap_search_s(ld, search_base, LDAP_SCOPE_SUBTREE, filter,
	    attrs, FALSE, &res)) != LDAP_SUCCESS &&
	    rc != LDAP_SIZELIMIT_EXCEEDED && rc != LDAP_TIMELIMIT_EXCEEDED) {
		ldap_perror(ld, "ldap_search_s");
		return(NULL);
	}
	if ((matches = ldap_count_entries(ld, res)) < 0) {
		ldap_perror(ld, "ldap_count_entries");
		return(NULL);
	}
	else if (matches == 1) {
		/*
		 *  If we only found one match, and it was a fuzzy match, let
		 *  the user know.
		 */
		if (tried_fuzzy)
			printf("  Found one approximate match for \"%s\"\n", who);
		if (ldap_search_s(ld, ldap_get_dn(ld, ldap_first_entry(ld, res)), LDAP_SCOPE_BASE, "objectClass=*", read_attrs, FALSE, &res) != LDAP_SUCCESS) {
			ldap_perror(ld, "ldap_search_s");
			return(NULL);
		}
		return(res);
	}
	/* that didn't work -- try an inexact match */
	else if (matches == 0) {
		sidx++;
		goto try_again;
	}

	/*
	 *  If we are here, it means that we got back multiple answers.
	 *  This can happen with both fuzzy and exact matches.
	 */
	if ((ld->ld_errno == LDAP_TIMELIMIT_EXCEEDED) || (ld->ld_errno == LDAP_SIZELIMIT_EXCEEDED)) {
		if (verbose) {
			printf("  Your query was too general and a limit was exceeded.  The results listed\n");
			printf("  are not complete.  You may want to try again with a more refined query.\n\n");
		}
		else
			printf("  Time or size limit exceeded.  Partial results follow.\n\n");
	}
	printf("  %1d names matched \"%s\".\n", matches, who);
	for (;;) {
		printf("  Do you wish to see a list of names? ");
		fflush(stdout);
		(void) memset(response, 0, sizeof(response));
		fetch_buffer(response, sizeof(response), stdin);
		switch (response[0]) {
		case 'n' :
		case 'N' :
		case '\0' :
		case '\n' :
			return(NULL);
			/* NOTREACHED */
		case 'y' :
		case 'Y' :
			print_list(res, namelist, &matches);
			if (num_picked == 0)
				choice = pick_one(matches);
			else
				choice = num_picked;
			num_picked = 0;
			if (choice >= 0)
				strcpy(name, namelist[choice]);
			/*
			 *  Now free up all of the pointers malloc'ed in
			 *  namelist.
			 */
			for (i = 0; namelist[i] != NULL; i++)
				free(namelist[i]);
			if (choice < 0)
				return(NULL);
			if (ldap_search_s(ld, name, LDAP_SCOPE_BASE, "objectClass=*", read_attrs, FALSE, &res) != LDAP_SUCCESS) {
				ldap_perror(ld, "ldap_search_s");
				return(NULL);
			}
			return(res);
			/* NOTREACHED */
		default :
			printf("  What?\n");
			break;
		}
	}
	/* NOTREACHED */
}

pick_one(i)
int i;
{
	int n;
	char user_pick[RESP_SIZE];

#ifdef DEBUG
	if (debug & D_TRACE)
		printf("->pick_one(%d)\n", i);
#endif
	
	/* make the user pick an entry */
	for (;;) {
		printf("  Enter the number of the name you want or Q to quit: ");
		fflush(stdout);
		fetch_buffer(user_pick, sizeof(user_pick), stdin);
		if (user_pick[0] == 'q' || user_pick[0] == 'Q')
			return(-1);
		n = atoi(user_pick);
		if ((n > 0) && (n <= i))
			return(n);
		printf("  Invalid response\n");
	}
	/* NOTREACHED */
}

print_list(list, names, matches)
LDAPMessage *list;
char *names[];
int *matches;
{
	char **rdns;
	extern int lpp;
	char resp[RESP_SIZE];
	register LDAPMessage *ep;
	register int i = 1;
	register int rest = 4;		/* 4, not 1 */

#ifdef DEBUG
	if (debug & D_TRACE)
		printf("->print_list(%x, %x, %x)\n", list, names, matches);
#endif
	/* print a list of names from which the user will select */
	for (ep = ldap_first_entry(ld, list); ep != NULL; ep = ldap_next_entry(ld, ep)) {
		
		names[i] = ldap_get_dn(ld, ep);
		rdns = ldap_explode_dn(names[i], TRUE);
		printf(" %3d. %s\n", i, *rdns);
		ldap_value_free(rdns);
		i++;
		if ((rest++ > (lpp - 1)) && (i < *matches)) {
again:
			printf("  More? ");
			fflush(stdout);
			fetch_buffer(resp, sizeof(resp), stdin);
			if ((resp[0] == 'n') || (resp[0] == 'N'))
				break;
			else if ((num_picked = atoi(resp)) != 0) {
				if (num_picked < i)
					break;
				else
					goto again;
			}
			rest = 1;
		}
	}
	*matches = i - 1;
	names[i] = NULL;
	return;
}

#ifdef UOFM

char * fetch_update_setting(who)
char *who;
{
	int setting;			/* 1 if TRUE, 0 if FALSE */
	BerElement *cookie;		/* it's magic, you know */
	LDAPMessage *result;		/* from the search below */
	register LDAPMessage *ep;	/* entry pointer */
	register char *ap, **vp;	/* for parsing the result */
	static char *attributes[] = { "noBatchUpdates", NULL };

#ifdef DEBUG
        if (debug & D_TRACE)
		printf("->fetch_update_setting(%s)\n", who);
#endif

	if (ldap_search_s(ld, who, LDAP_SCOPE_BASE, "objectClass=*", attributes, FALSE, &result) != LDAP_SUCCESS) {
		if (ld->ld_errno == LDAP_NO_SUCH_ATTRIBUTE)
			return("FALSE");
		ldap_perror(ld, "ldap_search_s");
		return(NULL);
	}

	/*
	 *  We did a read on one name and only asked for one attribute.
	 *  There's no reason to loop through any of these structures.
	 *
	 *  We also don't handle the case where noBatchUpdates has more
	 *  than one value.  That should never happen anyhow.
	 *
	 *  If ldap_first_attribute() returns NULL, then this entry did
	 *  not have a noBatchUpdates attribute.
	 */
	ep = ldap_first_entry(ld, result);
	if ((ap = ldap_first_attribute(ld, ep, &cookie)) == NULL)
		return("FALSE");
	if (!strcasecmp(ap, "noBatchUpdates")) {
		vp = (char **) ldap_get_values(ld, ep, ap);
		if (!strcasecmp(*vp, "TRUE"))
			setting = 1;
		else if (!strcasecmp(*vp, "FALSE"))
			setting = 0;
		else {
			fprintf(stderr, "  Got garbage -> [%s]\n", *vp);
			ldap_value_free(vp);
			return(NULL);
		}
		ldap_value_free(vp);
		if (setting == 1)
			return("TRUE");
		else
			return("FALSE");
	}
	else {
		fprintf(stderr, "  Got garbage -> [%s]\n", ap);
		return(NULL);
	}
}

#endif


find_all_subscribers(sub, group)
char *sub[];
char *group;
{
	int count;
	LDAPMessage *result;
	static char *attributes[] = { "cn", NULL };
	char filter[BUFSIZ];
	register LDAPMessage *ep;
	register int i = 0;

#ifdef DEBUG
	if (debug & D_TRACE)
		printf("->find_all_subscribers(%x, %s)\n", sub, group);
#endif

	sprintf(filter, "%s=%s", "memberOfGroup", group);
	if (ldap_search_s(ld, search_base, LDAP_SCOPE_SUBTREE, filter, attributes, FALSE, &result) != LDAP_SUCCESS) {
		if (ld->ld_errno == LDAP_NO_SUCH_ATTRIBUTE)
			return(0);
		ldap_perror(ld, "ldap_search_s");
		return(0);
	}
	count = ldap_count_entries(ld, result);
	if (count < 1)
		return(0);

	for (ep = ldap_first_entry(ld, result); ep != NULL; ep = ldap_next_entry(ld, ep)) {
		sub[i++] = strdup(ldap_get_dn(ld, ep));
#ifdef DEBUG
		if (debug & D_PARSE)
			printf("sub[%d] = %s\n", i - 1, sub[i - 1]);
#endif
	}
	sub[i] = NULL;
	ldap_msgfree(result);
	return(count);
}



char ** fetch_attrs(who, what)
char *who, *what;
{
	BerElement *cookie;		/* it's magic, you know */
	LDAPMessage *result;		/* from the search below */
	register LDAPMessage *ep;	/* entry pointer */
	register char *ap;		/* for parsing the result */
	static char *attributes[] = { NULL, NULL };

#ifdef DEBUG
        if (debug & D_TRACE)
		printf("->fetch_attrs(%s, %s)\n", who, what);
#endif

	attributes[0] = what;
	if (ldap_search_s(ld, who, LDAP_SCOPE_BASE, "objectClass=*", attributes, FALSE, &result) != LDAP_SUCCESS) {
		if (ld->ld_errno == LDAP_NO_SUCH_ATTRIBUTE)
			return((char **) NULL);
		ldap_perror(ld, "ldap_search_s");
		return((char **) NULL);
	}

	/*
	 *  We did a read on one name and only asked for one attribute.
	 *  There's no reason to loop through any of these structures.
	 *
	 *  We also don't handle the case where 'what' has more
	 *  than one value.
	 *
	 */
	ep = ldap_first_entry(ld, result);
	if ((ap = ldap_first_attribute(ld, ep, &cookie)) == NULL)
		return((char **) NULL);
	return((char **) ldap_get_values(ld, ep, ap));
}
