/*
 * Copyright (c) 1991, 1992 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <lber.h>
#include <ldap.h>
#include "ud.h"
#ifdef KERBEROS
#include <sys/types.h>
#include <krb.h>
#endif

char me[BUFSIZ];		/* the user's set name */
char password[512];		/* password entered my user */
char bound_as[BUFSIZ];		/* the user's bound name */
int verbose;			/* verbosity indicator */
extern LDAP *ld;		/* our LDAP descriptor */

extern char *mygetpass();	/* getpass() passwds are too short */

#ifdef DEBUG
extern int debug;		/* debug flag */
#endif

#ifdef KERBEROS
static char tktpath[20];	/* ticket file path */
static int kinit();
static int valid_tgt();
#endif

bind_info()
{
	printf("  Bound as \"%s\"\n", bound_as);
}

auth(who, best_guess)
char *who, *best_guess;
{
	int rc;			/* return code from ldap_bind() */
	char *passwd = NULL;	/* returned by mygetpass() */
	char *dn;		/* distinghished name from ldap_get_dn() */
	char **rdns;		/* for fiddling with the DN */
	int authmethod;
#ifdef KERBEROS
	char **krbnames;	/* for kerberos names */
	int kinited, ikrb;
	char buf[5];
	extern int krb_debug;
#endif
	LDAPMessage *mp;	/* returned from find() */
	static char prompt[BUFSIZ];	/* place for us to sprintf the prompt */
	static char name[BUFSIZ];	/* place to store the user's name */
	extern char *bound_dn;		/* DN of bound user */
	extern struct entry2 Entry;	/* look here for a name if needed */
	extern LDAPMessage *find();	/* for looking up 'name' */
	extern char *search_base;	/* for printing later */
	extern char *default_bind_object;	/* bind as this on failure */
	extern void printbase();	/* used to pretty-print a base */

#ifdef DEBUG
	if (debug & D_TRACE)
		fprintf(stderr, "auth(%s,%s)\n", who, best_guess);
#endif

	if (who == NULL) {
		/*
		 *  The user needs to bind.  If <who> is not specified, we
		 *  can make some guesses.  If the user has used the 'iam'
		 *  command, then try the name s/he set.  If not, try the
		 *  name retrieved with the last 'find' operation.  If no
		 *  such command has been issued, we don't have any guess,
		 *  and the user will have to supply a name.
		 */
		if (me[0] != '\0')
			who = me;
		else if (Entry.name != NULL)
			who = Entry.name;
		else if (best_guess != NULL)
			who = best_guess;
		do {
			if (who == NULL) {
				printf("  What is your name? ");
				fflush(stdout);
				fetch_buffer(name, sizeof(name), stdin);
				if (name[0] != '\0')
					who = name;
			}
			else {
				printf("  Is your name \"%s\"? ", who);
				fflush(stdout);
				fetch_buffer(name, sizeof(name), stdin);
				if ((name[0] != '\0') && strncasecmp(name, "yes", strlen(name)))
					who = NULL;
					/* and loop again */
			}
		} while (who == NULL);
	}

#ifdef DEBUG
	if (debug & D_AUTHENTICAT)
		printf("  Authenticating as \"%s\"\n", who);
#endif

	/*
	 *  Bail out if the name is bogus.  If not, strip off the junk
	 *  at the start of the DN, build a prompt, and get a password 
	 *  from the user.  Then perform the ldap_bind().
	 */
	if ((mp = find(who)) == NULL) {
		(void) ldap_msgfree(mp);
		printf("  I could not find \"%s\" in the Directory.\n", who);
		printf("  I used a search base of ");
		printbase("", search_base);
		printf("\n");
#ifdef DEBUG
		if (debug & D_AUTHENTICAT)
			printf("  Could not find \"%s\"\n", who);
#endif
		return(-1);
	}

	/*
	 *  Fill in the Entry structure.  May be handy later.
	 */
	(void) parse_answer(mp);

	dn = ldap_get_dn(ld, ldap_first_entry(ld, mp));
	rdns = ldap_explode_dn(dn, TRUE);

#ifdef KERBEROS
	/*
	 * First, if the user has a choice of auth methods, ask which
	 * one they want to use.  if they want kerberos, ask which
	 * krbname they want to bind as.
	 */

	if ( (krbnames = ldap_get_values( ld, mp, "krbName" )) != NULL ) {
		int 	choice, hassimple;

		hassimple = (ldap_compare_s( ld, dn, "userPassword", "x" ) ==
		    LDAP_COMPARE_FALSE);

		/* if we're running as a server (e.g., out of inetd) */
		if ( ! isatty( 1 ) ) {
			strcpy( tktpath, "/tmp/ud_tktXXXXXX" );
			mktemp( tktpath );
			krb_set_tkt_string( tktpath );
		}

		if ( hassimple ) {
			printf("  Which password would you like to use?\n");
			printf("    1 -> X.500 password\n");
#ifdef UOFM
			printf("    2 -> Uniqname (Kerberos) password\n");
#else
			printf("    2 -> Kerberos password\n");
#endif

			do {
				printf("  Enter 1 or 2: ");
				fflush(stdout);

				fetch_buffer(buf, sizeof(buf), stdin);
				choice = atoi(buf);
			} while (choice != 1 && choice != 2);

			authmethod = (choice == 1 ? LDAP_AUTH_SIMPLE :
			    LDAP_AUTH_KRBV4);
		} else {
			authmethod = LDAP_AUTH_KRBV4;
		}

		if ( authmethod == LDAP_AUTH_KRBV4 )
			kinited = valid_tgt( krbnames );
	} else {
		authmethod = LDAP_AUTH_SIMPLE;
	}

	/*
	 * if they are already kinited, we don't need to ask for a 
	 * password.
	 */

	if ( authmethod == LDAP_AUTH_KRBV4 ) {
		if ( ! kinited ) {
			if ( krbnames[1] != NULL ) {
				int	i;

				/* ask which one to use */
#ifdef UOFM
				printf("  Which Kerberos (uniqname) name would you like to use?\n");
#else
				printf("  Which Kerberos name would you like to use?\n");
#endif
				for ( i = 0; krbnames[i] != NULL; i++ ) {
					printf( "    %d -> %s\n", i + 1,
					    krbnames[i] );
				}
				do {
					printf("  Enter a number between 1 and %d: ", i );
					fflush( stdout );

					fetch_buffer(buf, sizeof(buf), stdin);
					ikrb = atoi(buf) - 1;
				} while ( ikrb > i - 1 || ikrb < 0 );
			} else {
				ikrb = 0;
			}

			/* kinit */
			if ( kinit( krbnames[ikrb] ) != 0 ) {
				(void) ldap_value_free(rdns);
				(void) ldap_value_free(krbnames);
				return(-1);
			}
			(void) ldap_value_free(krbnames);
		}
	} else {
#endif
		(void) ldap_msgfree(mp);
		authmethod = LDAP_AUTH_SIMPLE;
		sprintf(prompt, "  Enter your X.500 password: ");
		do {
			passwd = mygetpass(prompt);
		} while (passwd != NULL && *passwd == '\0');
		if (passwd == NULL) {
			(void) ldap_value_free(rdns);
			return(0);
		}
#ifdef KERBEROS
	}
#endif
	ldap_flush_cache( ld );
	rc = ldap_bind_s(ld, dn, passwd, authmethod);
	if (rc != LDAP_SUCCESS) {
		if (ld->ld_errno == LDAP_NO_SUCH_ATTRIBUTE)
			fprintf(stderr, "  Entry has no password\n");
		else if (ld->ld_errno == LDAP_INVALID_CREDENTIALS)
			fprintf(stderr, "  Password does not match\n");
		else
			ldap_perror(ld, "ldap_bind_s" );
		(void) ldap_value_free(rdns);
		(void) ldap_bind_s(ld, default_bind_object,
			 (char *) UD_PASSWD, LDAP_AUTH_SIMPLE);
		(void) strcpy(bound_as, DEFAULT_BOUND_AS);
		if (default_bind_object == NULL)
			bound_dn = NULL;
		else
			bound_dn = strdup(default_bind_object);
		if (verbose)
			printf("  Authentication failed.\n\n");
		return(-1);
	}
	else if (verbose)
		printf("  Authentication successful.\n\n");
	else
		printf("\n");
	(void) strcpy(bound_as, *rdns);
	bound_dn = strdup(dn);
	if (passwd != NULL)
		(void) strcpy(password, passwd);
	(void) ldap_value_free(rdns);
	return(0);
}

iam(who)
char *who;
{
	if (!strcmp(who, "me"))
		printf("  Of course you are.\n");
	else {
		(void) strcpy(me, who);
		print_my_name();
	}
}

print_my_name()
{
	if (me[0] == '\0')
		printf("  I do not know who you are.  Use IAM to tell me.\n");
	else
		printf("  You are \"%s\".\n", me);
}

#ifdef KERBEROS

#define FIVEMINS	( 5 * 60 )
#define TGT		"krbtgt"

str2upper( s )
    char	*s;
{
	char	*p;

	for ( p = s; *p != '\0'; ++p ) {
		if ( islower( *p )) {
			*p = toupper( *p );
		}
	}
}


static valid_tgt( names )
    char	**names;
{
	int		i;
	char		name[ ANAME_SZ ], inst[ INST_SZ ], realm[ REALM_SZ ];
	CREDENTIALS	cred;

	for ( i = 0; names[i] != NULL; i++ ) {
		if ( kname_parse( name, inst, realm, names[i] ) != KSUCCESS ) {
			fprintf( stderr, "Bad format for krbName %s\n",
			    names[i] );
			fprintf( stderr, "Contact x500@umich.edu\n" );
			return( 0 );
		}

#ifdef AFSKERBEROS
		/*
		 * realm must be uppercase for krb_ routines
		 */
		str2upper( realm );
#endif /* AFSKERBEROS */

		/*
		* check ticket file for a valid ticket granting ticket
		* my check is: have ticket granting ticket and it is good for
		* at least 5 more minutes
		*/
		if ( krb_get_cred( TGT, realm, realm,
		    &cred ) == KSUCCESS && time( 0 ) + FIVEMINS <
		    cred.issue_date + (u_char)cred.lifetime * FIVEMINS ) {
			return( 1 );
		}
	}

	return( 0 );
}

static char *kauth_name;

/*ARGSUSED*/
int
krbgetpass( user, inst, realm, pw, key )
    char *user, *inst, *realm, *pw;
    C_Block key;
{
	char	*p, lcrealm[ REALM_SZ ], prompt[256], *passwd;

#ifdef UOFM
	sprintf(prompt, "  Enter the Uniqname password (same as Kerberos password)\n  for %s: ", kauth_name );
#else
	sprintf(prompt, "  Enter Kerberos password for %s: ", kauth_name );
#endif
	do {
		passwd = mygetpass(prompt);
	} while (passwd != NULL && *passwd == '\0');
	if (passwd == NULL) {
		return(-1);
	}

#ifdef AFSKERBEROS
	strcpy( lcrealm, realm );
	for ( p = lcrealm; *p != '\0'; ++p ) {
		if ( isupper( *p )) {
			*p = tolower( *p );
		}
	}

	ka_StringToKey( passwd, lcrealm, key );
#else /* AFSKERBEROS */
	string_to_key( passwd, key );
#endif /* AFSKERBEROS */

	return( 0 );
}

static kinit( kname )
    char	*kname;
{
	int	rc;
	char	name[ ANAME_SZ ], inst[ INST_SZ ], realm[ REALM_SZ ];

	kauth_name = kname;

	if ( kname_parse( name, inst, realm, kname ) != KSUCCESS ) {
		fprintf( stderr, "Bad format for krbName %s\n",
		    kname );
		fprintf( stderr, "Contact x500@umich.edu\n" );
		return( -1 );
	}

#ifdef AFSKERBEROS
	/*
	 * realm must be uppercase for krb_ routines
	 */
	str2upper( realm );
#endif /* AFSKERBEROS */

	rc = krb_get_in_tkt( name, inst, realm, TGT, realm,
	    DEFAULT_TKT_LIFE, krbgetpass, NULL, NULL );

	if ( rc != KSUCCESS ) {
		switch ( rc ) {
		case SKDC_CANT:
			fprintf( stderr, "Can't contact Kerberos server for %s\n", realm );
			break;
		default:
			fprintf( stderr, "%s: %s\n", name, krb_err_txt[ rc ] );
			break;
		}
		return( -1 );
	}

	return( 0 );
}

destroy_tickets()
{
	if ( *tktpath != '\0' ) {
		unlink( tktpath );
	}
}
#endif
