/*
 *  Copyright (c) 1993 Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  getfilter.c -- optional add-on to libldap
 */

#ifndef lint 
static char copyright[] = "@(#) Copyright (c) 1993 Regents of the University of Michigan.\nAll rights reserved.\n";
#endif

#include "lber.h"
#include "ldap.h"
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#ifdef MACOS
#include <stdlib.h>
#include "macos.h"
#else /* MACOS */
#ifdef DOS
#include "msdos.h"
#else /* DOS */
#include <sys/errno.h>
#endif /* DOS */
#endif /* MACOS */
#if defined( MACOS ) || defined( DOS )
#include "regex.h"
#endif /* MACOS or DOS */

#ifdef NEEDPROTOS
static char *next_token( char **sp );
static void build_filter( char *pattern, char *attr, char *value,
	char **valwords, char *buf, int buflen );
static int break_into_words( char *str, char *delims, char ***wordsp );
#else /* NEEDPROTOS */
static char *next_token();
static void build_filter();
static int break_into_words();
#endif /* NEEDPROTOS */

#if !defined( MACOS ) && !defined( DOS )
extern int	errno;
#endif

#define FILT_MAX_LINE_LEN	1024
#define FILT_MAX_TOKEN_LEN	128

LDAPFiltDesc *
ldap_init_getfilter( fname )
    char		*fname;
{
    LDAPFiltDesc	*lfdp;
    LDAPFiltList	*flp, *nextflp;
    LDAPFiltInfo	*fip, *nextfip;
    FILE		*fp;
    char		linebuf[ FILT_MAX_LINE_LEN ];
    char		*p, *tag, *tok[ 5 ];
    int			tokcnt, i;


    if ( fname == NULL ) {
	return( NULL );
    }

    if (( lfdp = (LDAPFiltDesc *)calloc( 1, sizeof( LDAPFiltDesc))) == NULL ) {
	return( NULL );
    }

    if (( fp = fopen( fname, "r" )) == NULL ) {
	free( lfdp );
	return( NULL );
    }

    flp = nextflp = NULL;
    fip = NULL;
    tag = NULL;

    while ( fgets( linebuf, sizeof( linebuf ), fp ) != NULL ) {
	linebuf[ strlen( linebuf ) - 1 ] = '\0';
	p = linebuf;

	if ( *p == '#' || ( tok[ 0 ] = next_token( &p )) == NULL ) {
	    /* skip comments and blank lines */
	    nextflp = NULL;
	    continue;
	}

	for ( tokcnt = 1;
		tokcnt < 5 && ( tok[ tokcnt ] = next_token( &p )) != NULL;
		++tokcnt ) {
	    ;
	}

	switch( tokcnt ) {
	case 1:		/* tag line */
	    if ( tag != NULL ) {
		free( tag );
	    }
	    tag = tok[ 0 ];
	    break;
	case 4:
	case 5:		/* start of filter info. list */
	    if (( nextflp = (LDAPFiltList *)malloc( sizeof( LDAPFiltList )))
		    == NULL ) {
		ldap_getfilter_free( lfdp );
		return( NULL );
	    }
	    nextflp->lfl_tag = strdup( tag );
	    nextflp->lfl_pattern = tok[ 0 ];
	    if ( re_comp( nextflp->lfl_pattern ) != NULL ) {
#ifndef NO_USERINTERFACE
		ldap_getfilter_free( lfdp );
		fprintf( stderr, "bad regular expresssion %s\n",
			nextflp->lfl_pattern );
#if !defined( MACOS ) && !defined( DOS )
		errno = EINVAL;
#endif
#endif /* NO_USERINTERFACE */
		return( NULL );
	    }
		
	    nextflp->lfl_delims = tok[ 1 ];
	    nextflp->lfl_ilist = NULL;
	    nextflp->lfl_next = NULL;
	    if ( flp == NULL ) {	/* first one */
		lfdp->lfd_filtlist = nextflp;
	    } else {
		flp->lfl_next = nextflp;
	    }
	    flp = nextflp;
	    fip = NULL;
	    for ( i = 2; i < 5; ++i ) {
		tok[ i - 2 ] = tok[ i ];
	    }
	    /* fall through */

	case 2:
	case 3:		/* filter, desc, and optional search scope */
	    if ( nextflp != NULL ) { /* add to info list */
		if (( nextfip = (LDAPFiltInfo *)malloc( sizeof( LDAPFiltInfo )))
			== NULL ) {
		    ldap_getfilter_free( lfdp );
		    return( NULL );
		}
		if ( fip == NULL ) {	/* first one */
		    nextflp->lfl_ilist = nextfip;
		} else {
		    fip->lfi_next = nextfip;
		}
		fip = nextfip;
		nextfip->lfi_filter = tok[ 0 ];
		nextfip->lfi_desc = tok[ 1 ];
		if ( tok[ 2 ] != NULL ) {
		    if ( strcasecmp( tok[ 2 ], "subtree" ) == 0 ) {
			nextfip->lfi_scope = LDAP_SCOPE_SUBTREE;
		    } else if ( strcasecmp( tok[ 2 ], "onelevel" ) == 0 ) {
			nextfip->lfi_scope = LDAP_SCOPE_ONELEVEL;
		    } else if ( strcasecmp( tok[ 2 ], "base" ) == 0 ) {
			nextfip->lfi_scope = LDAP_SCOPE_BASE;
		    } else {
			ldap_getfilter_free( lfdp );
#if !defined( MACOS ) && !defined( DOS )
			errno = EINVAL;
#endif
			return( NULL );
		    }
		} else {
		    nextfip->lfi_scope = LDAP_SCOPE_SUBTREE;	/* default */
		}
	    }
	    break;

	default:
	    ldap_getfilter_free( lfdp );
#if !defined( MACOS ) && !defined( DOS )
	    errno = EINVAL;
#endif
	    return( NULL );
	}
    }

    return( lfdp );
}


void
ldap_getfilter_free( lfdp )
    LDAPFiltDesc	*lfdp;
{
    LDAPFiltList	*flp, *nextflp;
    LDAPFiltInfo	*fip, *nextfip;

    for ( flp = lfdp->lfd_filtlist; flp != NULL; flp = nextflp ) {
	for ( fip = flp->lfl_ilist; fip != NULL; fip = nextfip ) {
	    nextfip = fip->lfi_next;
	    free( fip->lfi_filter );
	    free( fip->lfi_desc );
	    free( fip );
	}
	nextflp = flp->lfl_next;
	free( flp->lfl_pattern );
	free( flp->lfl_delims );
	free( flp->lfl_tag );
	free( flp );
    }

    if ( lfdp->lfd_curfip != NULL ) {
	free( lfdp->lfd_curfip );
    }
    if ( lfdp->lfd_curvalcopy != NULL ) {
	free( lfdp->lfd_curvalcopy );
    }
    if ( lfdp->lfd_curvalwords != NULL ) {
	free( lfdp->lfd_curvalwords );
    }

    free( lfdp );
}


LDAPFiltInfo *
ldap_getfirstfilter( lfdp, tagpat, value )
    LDAPFiltDesc	*lfdp;
    char		*tagpat;
    char		*value;
{
    LDAPFiltList	*flp;

    if ( lfdp->lfd_curvalcopy != NULL ) {
	free( lfdp->lfd_curvalcopy );
	free( lfdp->lfd_curvalwords );
    }

    lfdp->lfd_curval = value;
    lfdp->lfd_curfip = NULL;

    for ( flp = lfdp->lfd_filtlist; flp != NULL; flp = flp->lfl_next ) {
	if ( re_comp( tagpat ) == NULL && re_exec( flp->lfl_tag ) == 1
		&& re_comp( flp->lfl_pattern ) == NULL
		&& re_exec( lfdp->lfd_curval ) == 1 ) {
	    lfdp->lfd_curfip = flp->lfl_ilist;
	    break;
	}
    }

    if ( lfdp->lfd_curfip == NULL ) {
	return( NULL );
    }

    if (( lfdp->lfd_curvalcopy = strdup( value )) == NULL ) {
	return( NULL );
    }

    if ( break_into_words( lfdp->lfd_curvalcopy, flp->lfl_delims,
		&lfdp->lfd_curvalwords ) < 0 ) {
	free( lfdp->lfd_curvalcopy );
	lfdp->lfd_curvalcopy = NULL;
	return( NULL );
    }

    return( ldap_getnextfilter( lfdp ));
}


LDAPFiltInfo *
ldap_getnextfilter( lfdp )
    LDAPFiltDesc	*lfdp;
{
    LDAPFiltInfo	*fip;

    if ( lfdp->lfd_curfip == NULL ) {
	return( NULL );
    }

    fip = lfdp->lfd_curfip;
    lfdp->lfd_curfip = lfdp->lfd_curfip->lfi_next;

    if ( fip == NULL ) {
	return( NULL );
    }

    build_filter( fip->lfi_filter, "", lfdp->lfd_curval, lfdp->lfd_curvalwords,
	    lfdp->lfd_filter, sizeof( lfdp->lfd_filter ));
    lfdp->lfd_retfi.lfi_filter = lfdp->lfd_filter;
    lfdp->lfd_retfi.lfi_desc = fip->lfi_desc;
    lfdp->lfd_retfi.lfi_scope = fip->lfi_scope;

    return( &lfdp->lfd_retfi );
}


static void
build_filter( pattern, attr, value, valwords, buf, buflen )
	char	*pattern;
	char	*attr;
	char	*value;
	char	**valwords;
	char	*buf;
	int	buflen;
{
	char	*p, *f;
	int	i, wordcount, wordnum, endwordnum;
	
	for ( wordcount = 0; valwords[ wordcount ] != NULL; ++wordcount ) {
	    ;
	}

	f = buf;
	
	for ( p = pattern; *p != '\0'; ++p ) {
	    if ( *p == '%' ) {
		++p;
		if ( *p == 'a' ) {
		    memcpy( f, attr, strlen( attr ));
		    f += strlen( attr );
		} else if ( *p == 'v' ) {
		    if ( isdigit( *(p+1))) {
			++p;
			wordnum = *p - '1';
			if ( *(p+1) == '-' ) {
			    ++p;
			    if ( isdigit( *(p+1))) {
				++p;
				endwordnum = *p - '1';	/* e.g., "%v2-4" */
				if ( endwordnum > wordcount - 1 ) {
				    endwordnum = wordcount - 1;
				}
			    } else {
				endwordnum = wordcount - 1;  /* e.g., "%v2-" */
			    }
			} else {
			    endwordnum = wordnum;	/* e.g., "%v2" */
			}

			for ( i = wordnum; i <= endwordnum; ++i ) {
			    if ( i > wordnum ) {  /* add blank btw words */
				*f++ = ' ';
			    }
			    memcpy( f, valwords[ i ], strlen( valwords[ i ] ));
			    f += strlen( valwords[ i ] );
			}
		    } else {
			memcpy( f, value, strlen( value ));
			f += strlen( value );
		    }
		} else {
		    *f++ = *p;
		}
	    } else {
		*f++ = *p;
	    }
		
	    if ( f - buf > buflen ) {	/* sanity check */
		--f;
		break;
	    }
	}
	
	*f = '\0';
}


static int
break_into_words( str, delims, wordsp )
	char	*str;		/* this is used to make up the words */
	char	*delims;
	char	***wordsp;
{
    char	*word, **words;
    int		count;
	
    if (( words = (char **)malloc( sizeof( char * ))) == NULL ) {
	return( -1 );
    }
    count = 0;
    words[ count ] = NULL;

    word = strtok( str, delims );
    while ( word != NULL ) {
	if (( words = (char **)realloc( words,
		( count + 2 ) * sizeof( char * ))) == NULL ) {
	    return( -1 );
	}

	words[ count ] = word;
	words[ ++count ] = NULL;
	word = strtok( NULL, delims );
    }
	
    *wordsp = words;
    return( count );
}


static char *
next_token( sp )
    char	**sp;
{
    int		in_quote = 0;
    char	*p, *t, tok[ FILT_MAX_TOKEN_LEN ];


    if ( **sp == '\0' ) {
	return( NULL );
    }

    p = *sp;

    while ( isspace( *p )) {		/* skip leading white space */
	++p;
    }

    if ( *p == '\0' ) {
	return( NULL );
    }

    if ( *p == '\"' ) {
	in_quote = 1;
	++p;
    }
    t = tok;

    for ( ;; ) {
	if ( *p == '\0' || ( isspace( *p ) && !in_quote )) {
	    *t++ = '\0';		/* end of token */
	    break;
	}

	if ( *p == '\"' ) {
	    in_quote = !in_quote;
	    ++p;
	} else {
	    *t++ = *p++;
	}
    }

    *sp = p;

    if ( t == tok ) {
	return( NULL );
    }

    return( strdup( tok ));
}
