/*
    This file is part of the FElt finite element analysis package.
    Copyright (C) 1993 Jason I. Gobat and Darren C. Atkinson

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*****************************************************************************
*
* File:		elements.c
* 
* Description:	contains various commonly used routines in formulating
*		element stiffness matrices
*
******************************************************************************/

# include <math.h>
# include "element.h"

/*****************************************************************************
*
* Function:	GaussPoints
*
* Description:	sets an array containing the appropriate Gauss points
*		for a given number of points for Gaussian quadrature
*
******************************************************************************/

unsigned GaussPoints (npoints, xpoints, weights)
   unsigned	npoints;
   double	**xpoints,
		**weights;
{
   static double  x[3][3] = {{0.0,0.0,0.0},
                             {-0.57735026918962,0.57735026918962,0.0},
                             {-0.77459666924148,0.0,0.77459666924148}};
   static double  w[3][3] = {{1.0,0.0,0.0},
                             {1.0,1.0,0.0},
                             {0.5555555555555,0.8888888888888,0.5555555555555}};

   if (npoints > 3 || npoints < 1)
      return 1;
   else {
      if (xpoints != NULL)
         *xpoints = x[npoints - 1];
      if (weights != NULL)
         *weights = w[npoints - 1];
      
      return 0;
   }
}

/*****************************************************************************
*
* Function:	PlaneStrainD
*
* Parameters:	element		an arbitrary planar plane strain element
*		number		global element number
*
* Return value:	a D matrix appropriate to the element
*
*****************************************************************************/

Matrix PlaneStrainD (element)
   Element	element;
{
   static Matrix	D = NullMatrix;
   static Material	prev_material = NULL;
   double 		poisson,
			factor;

   if (D == NullMatrix) {
      D = CreateMatrix (3,3);
 
      if (D == NullMatrix)
         Fatal ("allocation error creating element stiffness matrix");
   }

   if (element -> material != prev_material) {
      ZeroMatrix (D);

      poisson = element -> material -> nu;

      MatrixData (D) [1][1] = 1 - poisson;
      MatrixData (D) [1][2] = poisson; 
      MatrixData (D) [2][1] = poisson;
      MatrixData (D) [2][2] = 1 - poisson;
      MatrixData (D) [3][3] = (1 - 2*poisson)/2;

      if (1 - 2*poisson <= TINY) {
         error ("singularity in constitutive matrix for element %d",element -> number);
         return NULL;
      }

      factor = element -> material -> E / ((1 + poisson)*(1 - 2*poisson));
      ScaleMatrix (D,D,factor,0.0);

      prev_material = element -> material;
   }

   return D;
}

/*****************************************************************************
*
* Function:	PlaneStressD
*
* Parameters:	element		an arbitrary planar plane strain element
*		number		global element number
*
* Return value:	a D matrix appropriate to the element
*
*****************************************************************************/

Matrix PlaneStressD (element)
   Element	element;
{
   static Matrix	D = NullMatrix;
   static Material	prev_material = NULL;
   double 		poisson,
			factor;

   if (D == NullMatrix) {
      D = CreateMatrix (3,3);
 
      if (D == NullMatrix)
         Fatal ("allocation error creating element stiffness matrix");
   }

   if (element -> material != prev_material) {
      ZeroMatrix (D);

      poisson = element -> material -> nu;

      MatrixData (D) [1][1] = 1;
      MatrixData (D) [1][2] = poisson; 
      MatrixData (D) [2][1] = poisson;
      MatrixData (D) [2][2] = 1;
      MatrixData (D) [3][3] = (1 - poisson)/2;

      if (1 - poisson*poisson <= TINY) {
         error ("singularity in constitutive matrix for element %d",element -> number);
         return NullMatrix;
      }

      factor = element -> material -> E / (1 - poisson*poisson);
      ScaleMatrix (D,D,factor,0.0);

      prev_material = element -> material;
   }

   return D;
}

double ElementLength (element, coords)
   Element	element;
   unsigned 	coords;
{
   if (coords == 2) 
      return sqrt ((element -> node[2] -> x - element -> node[1] -> x)*
                   (element -> node[2] -> x - element -> node[1] -> x) +
                   (element -> node[2] -> y - element -> node[1] -> y)* 
                   (element -> node[2] -> y - element -> node[1] -> y));
   else
      return sqrt ((element -> node[2] -> x - element -> node[1] -> x)*
                   (element -> node[2] -> x - element -> node[1] -> x) +
                   (element -> node[2] -> y - element -> node[1] -> y)* 
                   (element -> node[2] -> y - element -> node[1] -> y) +
                   (element -> node[2] -> z - element -> node[1] -> z)* 
                   (element -> node[2] -> z - element -> node[1] -> z));
}
