/*
    This file is part of the FElt finite element analysis package.
    Copyright (C) 1993 Jason I. Gobat and Darren C. Atkinson

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/************************************************************************
 * File:	velvet.c						*
 *									*
 * Description:	This file contains the routines for the top level of	*
 *		the velvet application.					*
 ************************************************************************/

# include <stdio.h>
# include <unistd.h>
# include <X11/Intrinsic.h>
# include <X11/StringDefs.h>
# include <X11/Xresource.h>
# include <X11/Xaw/AsciiText.h>
# include <X11/Xaw/Command.h>
# include <X11/Xaw/Dialog.h>
# include <X11/Xaw/Form.h>
# include <X11/Xaw/Label.h>
# include <X11/Xaw/MenuButton.h>
# include <X11/Xaw/Reports.h>
# include <X11/Xaw/SimpleMenu.h>
# include <X11/Xaw/Sme.h>
# include <X11/Xaw/SmeBSB.h>
# include <X11/Xaw/Viewport.h>
# include "fe.h"
# include "Drawing.h"
# include "dialog.h"
# include "forms.h"
# include "procedures.h"
# include "problem.h"
# include "globals.h"
# include "definitions.h"
# include "vfe.h"
# include "error.h"
# include "check.xbm"
# include "question.xbm"
# include "menu16.xbm"
# include "truss.xbm"


static char *defaults [ ] = {
# include "velvet.ad.h"
NULL
};

/* some global values */

Boolean   changeflag = False;

Boolean	 snap_status = False;
Boolean	 grid_status = False;
Boolean  number_status = True;

Boolean	 edit_mode    = False;

float	 snap_size = 0.25;
float    grid_size = 1.0;
float	 xmin = 0.0;
float	 xmax = 10.0;
float	 ymin = 0.0;
float	 ymax = 10.0;

PanelId  last_command = -1;

/************************************************************************
 * Function:	 PanelCallback						*
 *									*
 * Parameters:	 widget     - callback widget (ignored)			*
 *		 clientData - pointer to panel id			*
 *		 callData   - callback data (ignored)			*
 *									*
 * Return value: none							*
 *									*
 * Calls:	 ???							*
 *									*
 * Called by:	 X11							*
 *									*
 * Global data:	 changeflag may be modified				*
 *									*
 * Description:	 Panelcallback is called when an action is taken on the	*
 *		 control panel.  The client data is a pointer to the	*
 *		 panel id which indicates what action is to be taken.	*
 ************************************************************************/

void PanelCallback (widget, clientData, callData)
    Widget    widget;
    XtPointer clientData;
    XtPointer callData;
{
    if (!edit_mode) {

       last_command = *(PanelId *) clientData;
       switch (*(PanelId *) clientData) {
       case AddNode:
           EditAddNode ();
           break;

       case DeleteNode:
	   EditDeleteNode ();
	   break;

       case EditNode:
           EditNodeNumber ();
           break;

       case MoveNodeId:
	   MoveNodeNumber ();
	   break;

       case AddElt:
           EditAddElement ();
	   break;

       case DeleteElt:
	   EditDeleteElement ();
	   break;

       case EditElement:
           EditElementNumber ();
           break;

       case SetType:
           SetActiveElementType ();
           break;

       case Generate:
           GenerateElements ();
	   break;

       case DefMaterial:
           GetMaterialDefinition ();
	   break;

       case SetMaterial:
           SetActiveMaterial ();
	   break;

       case EditMaterial:
	   EditMaterialInfo ();
	   break;

       case DelMaterial:
	   DeleteMaterial ();
	   break;

       case OpenMaterial:
           OpenMaterialFile ();
	   break;

       case SaveMaterial:
           WriteMaterialFile ();
	   break;

       case DefForce:
           GetForceDefinition ();
	   break;

       case SetForce:
           SetActiveForce ();
	   break;

       case EditForce:
	   EditForceInfo ();
	   break;

       case DelForce:
	   DeleteForce ();
	   break;

       case ApplyForce:
	   EditApplyForce ();
	   break;

       case DefLoad:
           GetDistributedDefinition ();
	   break;

       case SetLoad:
           SetActiveDistributed ();
	   break;

       case EditLoad:
	   EditDistributedInfo ();
	   break;

       case DelLoad:
	   DeleteDistributed ();
	   break;

       case ApplyLoad:
	   EditApplyLoad ( );
	   break;

       case DefConstraint:
           GetConstraintDefinition ();
	   break;

       case SetConstraint:
	   SetActiveConstraint ();
	   break;

       case EditConstraint:
	   EditConstraintInfo ();
	   break;

       case DelConstraint:
	   DeleteConstraint ();
	   break;

       case ApplyConstraint:
	   EditApplyConstraint ();
	   break;

       case Define:
	   DefineProblem ();
           break;

       case Solve:
	   CompactOperation (NULL);
           break;

       case Parameters:
           GetCanvasParameters ();
	   break;

       case Configure:
           GetConfiguration ();
	   break;

       case ZoomOut:
           ZoomAll ();
           break;

       case ZoomIn:
           ZoomStart ();
           break;
 
       case SaveXWD:
           SaveWidgetXWD ();
           break;

       case NumbersOnOff:
           ToggleNumberStatus ();
           break;

       case SnapOnOff:
           ToggleSnapStatus ();
           break;

       case GridOnOff:
           ToggleGridStatus ();
           break;

       case LineTool:
           ToolsDrawLine ();
	   break;

       case Circle:
           ToolsDrawCircle ();
	   break;

       case Arc:
           ToolsDrawArc ();
	   break;

       case Rectangle:
           ToolsDrawRectangle ();
	   break;

       case Polygon:
           ToolsDrawPolygon ();
	   break;

       case Text:
	   ToolsDrawText ();
           break;

       case DeleteTool:
	   ToolsDeleteFigure ();
           break;

       case NewId:
           StartNew ();
	   break;

       case Open:
           OpenFile ();
	   break;

       case Save:
	   SetShellTitle (error_dialog -> shellwidget, "Save Problem");
	   if (!filename [0])
	      WriteNamedFile ();
	   else {
              if (!WriteVFeltFile ()) {
                 WriteVelvetFile ();
	 	 changeflag = False;
	      }
	   }
	   break;

       case SaveAs:
           WriteNamedFile ();
	   break;

       case Restore:
           RestoreOriginal ();
	   break;

       case Exit:
           QuitVelvet ();
           break;

       case Info:
	   PopupDialog (info_dialog, "Velvet v1.0 by JIG and DCA", NULL, NULL);
	   break;

       case Quit:
       case Abort:
	   break;

       } /* end switch top of control panel */

    } /* end if !edit_mode */
  
    else {

       switch (*(PanelId *) clientData) {
       case SnapOnOff:
           ToggleSnapStatus ();
           break;

       case GridOnOff:
           ToggleGridStatus ();
           break;

       case NumbersOnOff:
           ToggleNumberStatus ();
           break;

       default:
           XBell (XtDisplay (toplevel), 20);

       } /* end switch bottom of control panel */
    }
}


/************************************************************************
 * Function:	SelectCallback
 ************************************************************************/

void SelectCallback (w, clientData, callData)
    Widget    w;
    XtPointer clientData;
    XtPointer callData;
{
    DrawingReport   *report;
    FigureAttributes attributes;
    Figure           figure;
    Node             node;
    Element          element;
    Drawn            drawn;


    report = (DrawingReport *) callData;

    if (report -> event -> type != ButtonPress)
	return;

    figure = DW_FindFigure (w, report -> unsnapped.x, report -> unsnapped.y);

    if (figure == NULL)
	return;

    DW_GetAttributes (w, figure, &attributes);
    if (attributes.user_data == NULL)
	return;

    node = (Node) attributes.user_data;
    element = (Element) attributes.user_data;
    drawn = (Drawn) node -> aux;

    if (drawn -> type == DrawnNode) {
        if (report -> event -> xbutton.button == 1)
	   GetNodeInformation (node);
	else if (report -> event -> xbutton.button == 2)
	   DoMoveNode (node, True);
        else if (report -> event -> xbutton.button == 3)
           EditNodeInfo (node);
    } else if (drawn -> type == DrawnElement) {
        if (report -> event -> xbutton.button == 1)
	   GetElementInformation (element);
        else if (report -> event -> xbutton.button == 3)
           EditElementInfo (element);
    }
}


static XrmOptionDescRec options [ ] = {
    {"-sensitive",    "*sensitiveMenus", XrmoptionNoArg, "True"},
    {"+sensitive",    "*sensitiveMenus", XrmoptionNoArg, "False"},
    {"-nodeColor",    "*nodeColor",      XrmoptionSepArg, NULL},
    {"-elementColor", "*elementColor",   XrmoptionSepArg, NULL},
    {"-toolColor",    "*toolColor",      XrmoptionSepArg, NULL},
    {"-labelFont",    "*labelFont",      XrmoptionSepArg, NULL},
    {"-toolFont",     "*toolFont",       XrmoptionSepArg, NULL},
};

struct resources {
    Boolean sensitive;
    String  nodecolor;
    String  elementcolor;
    String  toolcolor;
    String  labelfont;
    String  toolfont;
} appResources;

# define offset(field) XtOffsetOf (struct resources, field)
static XtResource Resources [ ] = {
{"sensitiveMenus", "SensitiveMenus", XtRBoolean, sizeof (Boolean),
    offset (sensitive), XtRImmediate, (XtPointer) True},
{"nodeColor", "NodeColor", XtRString, sizeof (String),
    offset (nodecolor), XtRImmediate, (XtPointer) "blue"},
{"elementColor", "ElementColor", XtRString, sizeof (String),
    offset (elementcolor), XtRImmediate, (XtPointer) "black"},
{"toolColor", "ToolColor", XtRString, sizeof (String),
    offset (toolcolor), XtRImmediate, (XtPointer) "red"},
{"labelFont", "LabelFont", XtRString, sizeof (String),
    offset (labelfont), XtRImmediate, (XtPointer) "5x8"},
{"toolFont", "ToolFont", XtRString, sizeof (String),
    offset (toolfont), XtRImmediate, (XtPointer) "fg-22"},
};
# undef offset


static void GetArgs ( )
{
    XtGetApplicationResources (toplevel, &appResources, Resources,
				XtNumber (Resources), NULL, 0);

    sensitive_menus = appResources.sensitive;
    node_color      = appResources.nodecolor;
    elt_color       = appResources.elementcolor;
    tool_color      = appResources.toolcolor;
    label_font      = appResources.labelfont;
    text_font       = appResources.toolfont;
}


/************************************************************************
 * Function:	 main							*
 *									*
 * Parameters:	 argc - number of command line arguments		*
 *		 argv - array of command line arguments			*
 *									*
 * Return value: 0 on success; nonzero on error				*
 *									*
 * Calls:	 ???							*
 *									*
 * Called by:	 none							*
 *									*
 * Global data:	 ???							*
 *									*
 * Description:	 Main is the startup function for the velvet program.	*
 ************************************************************************/

int main (argc, argv)
    int   argc;
    char *argv [ ];
{
    int		   i, j;
    char	   string [32];
    Arg		   arglist [14];
    Cardinal	   count;
    XtTranslations trans;
    Window	   window;
    Dimension	   width,
		   height;
    float	   xscale,
		   yscale; 


    /* Create the top level widgets. */


    toplevel = XtAppInitialize (&app_context, "Velvet", options,
		XtNumber (options), &argc, argv, defaults, NULL, 0);

    GetArgs ( );

    form     = XtCreateManagedWidget
		("form", formWidgetClass, toplevel, NULL, 0);

    control  = XtCreateManagedWidget
		("control", formWidgetClass, form, NULL, 0);

    viewport = XtCreateManagedWidget
		("viewport", viewportWidgetClass, form, NULL, 0);

    drawing  = XtCreateManagedWidget
		("drawing", drawingWidgetClass, viewport, NULL, 0);

    coord    = XtCreateManagedWidget
		("coord", labelWidgetClass, form, NULL, 0);

    bottom   = XtCreateManagedWidget
		("bottom", formWidgetClass, form, NULL, 0);

    status   = XtCreateManagedWidget
		("status", labelWidgetClass, bottom, NULL, 0);

    count = 0;
    XtSetArg (arglist [count], XtNeditType, XawtextEdit); count ++;

    entry    = XtCreateManagedWidget
		("entry", asciiTextWidgetClass, bottom, arglist, count);


    /* Focus all keyboard input to the text entry widget. */

    XtSetKeyboardFocus (form, entry);


    /* Create the bitmaps. */

    window = RootWindowOfScreen (XtScreen (toplevel));
    checkmark = XCreateBitmapFromData (XtDisplay(toplevel), window,
				        check_bits, check_width, check_height);

    question = XCreateBitmapFromData (XtDisplay (toplevel), window,
			        question_bits, question_width, question_height);

    menu16 = XCreateBitmapFromData (XtDisplay (toplevel), window,
				    menu16_bits, menu16_width, menu16_height);

    truss = XCreateBitmapFromData (XtDisplay (toplevel), window,
				      truss_bits, truss_width, truss_height);


    /* Create the control panel menus and buttons. */

    for (i = 0; i < XtNumber (panel); i ++) {
	if (panel [i].class == &menuButtonWidgetClass)
	    XtSetArg (arglist [0], XtNleftBitmap, menu16);
	else if (panel [i].class == &commandWidgetClass)
	    XtSetArg (arglist [0], XtNleftBitmap, question);
	else
	    XtSetArg (arglist [0], XtNlabel, "");

	panel [i].button = XtCreateManagedWidget
			   (panel [i].name, *panel [i].class, control, arglist, 1);

	if (panel [i].class == &menuButtonWidgetClass) {
	    sprintf (string, "%sMenu", panel [i].name);
	    panel [i].menu = XtCreatePopupShell
			     (string, simpleMenuWidgetClass, control, NULL, 0);

	    for (j = 0; j < panel [i].numentries; j ++) {
		panel [i].menuentry [j].widget = XtCreateManagedWidget
		 (panel [i].menuentry [j].name, smeBSBObjectClass,
		  panel [i].menu, NULL, 0);

		XtAddCallback (panel [i].menuentry [j].widget, XtNcallback,
		 PanelCallback, &panel [i].menuentry [j].id);
	    }

	} else if (panel [i].class == &commandWidgetClass)
	    XtAddCallback (panel [i].button, XtNcallback, PanelCallback,
	     &panel [i].menuentry [0].id);
    }

    quitbutton = XtCreateManagedWidget ("quit", commandWidgetClass,
                      control, NULL, 0);
    XtAddCallback (quitbutton, XtNcallback, QuitEdit, NULL);

    abortbutton = XtCreateManagedWidget ("abort", commandWidgetClass,
                      control, NULL, 0);
    XtAddCallback (abortbutton, XtNcallback, AbortEdit, NULL);
 


    /* Create the dialogs. */

    info_dialog    = CreateDialog (toplevel, "infod", Okay);
    error_dialog   = CreateDialog (toplevel, "errord", Okay);
    file_dialog    = CreateDialog (toplevel, "filed", Okay | Cancel);
    proceed_dialog = CreateDialog (toplevel, "proceedd", Okay | Cancel);
    qsave_dialog   = CreateDialog (toplevel, "qsaved", Yes | No | Cancel);

    XtSetArg (arglist [0], XtNicon, truss);
    XtSetValues (info_dialog -> dialogwidget, arglist, 1);


    /* Create the input forms */

    constraint_form = CreateToggleForm (toplevel, constraint_entries);
    material_list   = CreateListForm (toplevel, "Choose material:");
    element_list    = CreateListForm (toplevel, "Choose element type:");
    constraint_list = CreateListForm (toplevel, "Choose constraint:");
    force_list      = CreateListForm (toplevel, "Choose force:");
    load_list	    = CreateListForm (toplevel, "Choose load:");
    force_form      = CreateTextEntryForm (toplevel, force_entries);
    load_form	    = CreateTextEntryForm (toplevel, load_entries);
    node_form       = CreateTextEntryForm (toplevel, node_entries);
    element_form    = CreateTextEntryForm (toplevel, element_entries);
    trimesh_form    = CreateTextEntryForm (toplevel, trimesh_entries);
    material_form   = CreateTextEntryForm (toplevel, material_entries);
    canvas_form     = CreateTextEntryForm (toplevel, canvas_entries);
    config_form	    = CreateTextEntryForm (toplevel, config_entries);


    /* Create the output shell */

    outputShell = CreateOutputShell ( );

    /* Create the node and element information windows */

    CreateInformationShell (&nodeShell,"nodeShell",&nodewin,"nodewin");
    CreateInformationShell (&elementShell,"elementShell",
                            &elementwin,"elementwin");

    /* register the action table and set-up the translations */

    XtAppAddActions (app_context, actiontable, XtNumber (actiontable));
    trans = XtParseTranslationTable (default_translations);

    XtOverrideTranslations (entry, trans);


    /* Realize the top level widget */

    XtRealizeWidget (toplevel);

    count = 0;
    XtSetArg (arglist [count], XtNwidth, &width); count++;
    XtSetArg (arglist [count], XtNheight, &height); count++;
    XtGetValues (drawing, arglist, count);

    xscale = (float) width / (xmax - xmin);
    yscale = (float) height / (ymax - ymin);

    count = 0;
    XtSetArg (arglist [count], XtNcoordinates, coord); count++;
    XtSetArg (arglist [count], XtNxMin, Float2Arg(xmin)); count++;
    XtSetArg (arglist [count], XtNxMax, Float2Arg(xmax)); count++;
    XtSetArg (arglist [count], XtNyMin, Float2Arg(ymin)); count++;
    XtSetArg (arglist [count], XtNyMax, Float2Arg(ymax)); count++;
    XtSetArg (arglist [count], XtNyScale, Float2Arg(yscale)); count++;
    XtSetArg (arglist [count], XtNxScale, Float2Arg(xscale)); count++;
    XtSetArg (arglist [count], XtNgridSize, Float2Arg(grid_size)); count++;
    XtSetArg (arglist [count], XtNsnapSize, Float2Arg(snap_size)); count++;
    XtSetArg (arglist [count], XtNsnap, snap_status); count++;
    XtSetArg (arglist [count], XtNgrid, grid_status); count++; 
    XtSetValues (drawing, arglist, count);


    /* Initialize the element and node arrays */

    node = NULL;
    element = NULL;
    title = NULL;
    numnodes = 0;
    numelts = 0;

    /* check to see if the user started with a filename */

    SetShellTitle (error_dialog -> shellwidget, "Open Problem");
    StartNew ( );
    if (argc > 1) {
       if (access (argv[1], F_OK) == 0) {
          if (access (argv[1], R_OK) != 0)
             error ("File %s is not accessible, check file permissions.",
                    argv[1]);
          else {
	     strcpy (filename, argv [1]);
             ReadFeltFile (); 
             ReadVelvetFile ();
          }
       }
       else
	  strcpy (filename, argv [1]);
    }
    else 
       filename [0] = 0;
    
    UpdateTitle ();
    SetNormalMode ( );
    number_status = False;
    ToggleNumberStatus ( );

    /* Enter the main event loop */

    changeflag = False;
    XtAppMainLoop (app_context);

    return 0; /* NOTREACHED */
}
