/*
    This file is part of the FElt finite element analysis package.
    Copyright (C) 1993 Jason I. Gobat and Darren C. Atkinson

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

# include <stdio.h>
# include <ctype.h>
# include <string.h>
# include <stdlib.h>
# include <X11/Intrinsic.h>
# include "fe.h"
# include "forms.h"
# include "dialog.h"
# include "problem.h"
# include "globals.h"
# include "Drawing.h"
# include "vfe.h"
# include "util.h"
# include "objects.h"
# include "procedures.h"
# include "error.h"

static int count;


static int SetDistributedNames (item)
    Item item;
{
    if (active_distributed == (Distributed) item)
	active_distributed_number = count;

    distributed_names [count ++] = ((Distributed) item) -> name;
    return 0;
}


void GatherDistributedNames ( )
{
    int size;


    if ((size = TreeSize (problem.distributed_tree)) <= 0)
	return;

    count = 0;
    size *= sizeof (String);
    distributed_names = (String *) XtRealloc ((char *) distributed_names, size);
    (void) TreeSetIterator (problem.distributed_tree, SetDistributedNames);
    (void) TreeIterate (problem.distributed_tree);
}


void SetActiveDistributed ()
{
   int             	result, size;
   struct distributed	dummy;

   SetShellTitle (load_list -> shellwidget, "Set Active Load");
   if ((size = TreeSize (problem.distributed_tree)) <= 0) {
      error ("No loads defined.");
      return;
   }

   SetListItems (load_list, distributed_names, size);
   result = GetListFormValue (load_list, active_distributed_number);

   if (result > -1) {
      dummy.name = distributed_names [result];
      active_distributed = (Distributed) TreeSearch (problem.distributed_tree, &dummy);
      active_distributed_number = result;
      changeflag = True;
   }
}


void GetDistributedDefinition ()
{
   String 			result[4];
   extern EntryRecord		load_entries;
   unsigned			status;
   Distributed			distributed;
   int				num;
   char				buffer [80];
   Item			        found;
   unsigned			in_node1;
   unsigned			in_node2;
   double			in_mag1;
   double			in_mag2;
   Direction			dir;


   SetShellTitle (load_form -> shellwidget, "Define Load");
   status = GetEntryFormValues (load_form, load_entries, result);
   if (status) {
      if (strlen (result[0]) > 0)  {
	 distributed = CreateDistributed (XtNewString (result [0]), 2);

	 result [1] [0] = toupper (result [1] [0]);

	 if (!result [2] [0] && result [3] [0]) {
	    error ("First node and magnitude are absent.");
	    return;
	 }

	 if (!result [1] [0] || !strcmp (result [1], "X"))
	    dir = DirX;
	 else if (!strcmp (result [1], "Y"))
	    dir = DirY;
	 else if (!strcmp (result [1], "Z"))
	    dir = DirZ;
	 else if (!strncmp (result [1], "Parallel", 3))
	    dir = Parallel;
	 else if (!strncmp (result [1], "Perpendicular", 3))
	    dir = Perpendicular;
	 else {
	    error ("Illegal direction %s.", result [1]);
	    return;
	 }

	 if (result [2] [0]) {
	    num = sscanf (result [2], "%u%*[, ]%lf%s",&in_node1,&in_mag1,buffer);

	    if (num != 2) {
	       error ("Improper node / magnitude (1) syntax.");
	       DestroyDistributed (distributed);
	       return;
	    }
	 } else {
	    distributed -> nvalues = 0;
	    in_node1 = in_mag1 = 0;
	 }

	 if (result [3] [0]) {
	    num = sscanf (result [3], "%u%*[, ]%lf%s",&in_node2,&in_mag2,buffer);

	    if (num != 2) {
	       error ("Improper node / magnitude (2) syntax.");
	       DestroyDistributed (distributed);
	       return;
	    }
	 } else {
	    distributed -> nvalues = 1;
	    in_node2 = in_mag2 = 0;
	 }

	 distributed -> direction = dir;
	 distributed -> value [1].node = in_node1;
	 distributed -> value [1].magnitude = in_mag1;
	 distributed -> value [2].node = in_node2;
	 distributed -> value [2].magnitude = in_mag2;


	 found = TreeInsert (problem.distributed_tree, (Item) distributed);
	 if (found != (Item) distributed) {
	    error ("Load %s already exists.", distributed -> name);
	    DestroyDistributed (distributed);
	    return;
	 }

	 active_distributed = distributed;
	 GatherDistributedNames ( );
	 changeflag = True;
      } else
	 error ("No load name specified.");
   } 
}


void EditDistributedInfo ()
{
   int             	i, status, size;
   String		result[4];
   extern EntryRecord   load_entries;
   Distributed		distributed;
   static String        suggestion [4];
   static int           flag = 0;
   struct distributed	dummy;
   int			num;
   char			buffer [80];
   Item			found;
   unsigned		in_node1;
   unsigned		in_node2;
   double		in_mag1;
   double		in_mag2;
   Direction		dir;
   Boolean		new_load;


   if (!flag) {
	flag = 1;
	for (i = 0; i < 4; i ++)
	    suggestion [i] = (String) XtMalloc (sizeof (char) * 80);
   }

   SetShellTitle (load_list -> shellwidget, "Edit Load");
   if ((size = TreeSize (problem.distributed_tree)) <= 0) {
      error ("No loads defined.");
      return;
   }

   SetListItems (load_list, distributed_names, size);
   status = GetListFormValue (load_list, active_distributed_number);

   if (status == -1)
	return;

   dummy.name = distributed_names [status];
   distributed = (Distributed) TreeSearch (problem.distributed_tree, &dummy);

   strcpy (suggestion [0], distributed -> name);

   switch (distributed -> direction) {
   case DirX:
	strcpy (suggestion [1], "X");
	break;
   case DirY:
	strcpy (suggestion [1], "Y");
	break;
   case DirZ:
	strcpy (suggestion [1], "Z");
	break;
   case Parallel:
	strcpy (suggestion [1], "Parallel");
	break;
   case Perpendicular:
	strcpy (suggestion [1], "Perpendicular");
	break;
   default:
	strcpy (suggestion [1], "illegal direction");
   }

   sprintf (suggestion [2], "%u %g", distributed -> value [1].node,
				     distributed -> value [1].magnitude);
   sprintf (suggestion [3], "%u %g", distributed -> value [2].node,
				     distributed -> value [2].magnitude);
   FillTextBuffers (load_form, load_entries, suggestion);

   SetShellTitle (load_form -> shellwidget, "Edit Load");
   status = GetEntryFormValues (load_form, load_entries, result);

   new_load = False;

   if (status) {
      if (strlen (result[0]) > 0)  {
	 if (strcmp (result [0], distributed -> name)) {
	   distributed = CreateDistributed (XtNewString (result [0]), 2);
	   new_load = True;
	}

	 result [1] [0] = toupper (result [1] [0]);

	 if (!result [2] [0] && result [3] [0]) {
	    error ("First node and magnitude are absent.");
	    if (new_load == True)
		DestroyDistributed (distributed);
	    return;
	 }

	 if (!result [1] [0] || !strcmp (result [1], "X"))
	    dir = DirX;
	 else if (!strcmp (result [1], "Y"))
	    dir = DirY;
	 else if (!strcmp (result [1], "Z"))
	    dir = DirZ;
	 else if (!strncmp (result [1], "Parallel", 3))
	    dir = Parallel;
	 else if (!strncmp (result [1], "Perpendicular", 3))
	    dir = Perpendicular;
	 else {
	    error ("Illegal direction %s.", result [1]);
	    if (new_load == True)
		DestroyDistributed (distributed);
	    return;
	 }

	 if (result [2] [0]) {
	    num = sscanf (result [2], "%u%*[, ]%lf%s",&in_node1,&in_mag1,buffer);

	    if (num != 2) {
	       error ("Improper node / magnitude (2) syntax.");
	       if (new_load == True)
	           DestroyDistributed (distributed);
	       return;
	    }
	 } else {
	    distributed -> nvalues = 0;
	    in_node1 = in_mag1 = 0;
	 }

	 if (result [3] [0]) {
	    num = sscanf (result [3], "%u%*[, ]%lf%s",&in_node2,&in_mag2,buffer);

	    if (num != 2) {
	       error ("Improper node / magnitude syntax (2).");
	       if (new_load == True)
	          DestroyDistributed (distributed);
	       return;
	    }
	 } else {
	    distributed -> nvalues = 1;
	    in_node2 = in_mag2 = 0;
	 }

	 distributed -> direction = dir;
	 distributed -> value [1].node = in_node1;
	 distributed -> value [1].magnitude = in_mag1;
	 distributed -> value [2].node = in_node2;
	 distributed -> value [2].magnitude = in_mag2;


	 found = TreeInsert (problem.distributed_tree, (Item) distributed);
	 if (found != (Item) distributed) {
	    error ("Load %s already exists.", distributed -> name);
	    DestroyDistributed (distributed);
	    return;
	 }

	 active_distributed = distributed;
	 GatherDistributedNames ( );
	 changeflag = True;
      } else
	 error ("No load name specified.");
   }
}


static Distributed distributed;
static Boolean     first_time;

static int DistributedRef (item)
    Item item;
{
    unsigned i, j;
    unsigned status;
    Element  element = (Element) item;

    for (i = 1; i <= element -> numdistributed; i ++)
	if (element -> distributed [i] == distributed) {
	    if (first_time == True) {
		first_time = False;
		SetShellTitle (proceed_dialog -> shellwidget, "Delete Load");
		status = PopupDialog (proceed_dialog, "Load is still in use.  Proceed?", NULL, NULL);
		if (status != Okay)
		    return 1;
	    }

	    for (j = i + 1; j <= element -> numdistributed; j ++)
		element -> distributed [j - 1] = element -> distributed [j];

	    element -> numdistributed --;
	    i --;

	    if (displayed_element == element && XtIsRealized (elementShell) == True)
		GetElementInformation (displayed_element);
	}

    return 0;
}


void DeleteDistributed ()
{
   int             	result, size;
   struct distributed	dummy;

   SetShellTitle (load_list -> shellwidget, "Delete Load");
   if ((size = TreeSize (problem.distributed_tree)) <= 0) {
      error ("No loads defined.");
      return;
   }

   SetListItems (load_list, distributed_names, size);
   result = GetListFormValue (load_list, -1);

   if (result > -1) {
      first_time = True;
      dummy.name = distributed_names [result];
      distributed = (Distributed) TreeSearch (problem.distributed_tree, &dummy);
      (void) TreeSetIterator (problem.element_tree, DistributedRef);
      if (TreeIterate (problem.element_tree))
	  return;

      DestroyDistributed ((Distributed) TreeDelete (problem.distributed_tree, &dummy));
      if (active_distributed_number == result)
	active_distributed_number = -1;

      GatherDistributedNames ( );
      changeflag = True;
   }
}


void DoApplyLoad (element)
    Element element;
{
    static char message [80];


    if (element -> numdistributed == 3) {
	error ("Maximum number of loads applied.");
	return;
    }

    element -> distributed [++ element -> numdistributed] = active_distributed;

    if (XtIsRealized (elementShell))
	GetElementInformation (element);

    sprintf (message, "Applied to element %d.  Select element:", element -> number);
    ChangeStatusLine (message, True);
    changeflag = True;
}


void ApplyLoadCB (w, client_data, call_data)
    Widget    w;
    XtPointer client_data;
    XtPointer call_data;
{
    DrawingReport   *report;
    FigureAttributes attributes;
    Figure           figure;
    Drawn            drawn;
    Element          element;


    report = (DrawingReport *) call_data;

    if (report -> event -> type != ButtonPress)
	return;

    if (report -> event -> xbutton.button == 3)
	QuitEdit ( );

    if (report -> event -> xbutton.button != 1)
	return;

    figure = DW_FindFigure (w, report -> unsnapped.x, report -> unsnapped.y);

    if (figure == NULL)
	return;

    DW_GetAttributes (w, figure, &attributes);
    if (attributes.user_data == NULL)
	return;

    element = (Element) attributes.user_data;
    drawn = (Drawn) element -> aux;
    if (drawn -> type != DrawnElement)
	return;

    DoApplyLoad (element);
}


void ApplyLoadAP ( )
{
    char          *status;
    struct element dummy;
    Item           found;


    if ((status = GetTextNumber (&dummy.number)) != NULL)
	return;

    found = TreeSearch (problem.element_tree, (Item) &dummy);
    if (found == NULL) {
	error ("Element %d does not exist.", dummy.number);
	return;
    }

    DoApplyLoad (found);
}


void EditApplyLoad ( )
{
    SetShellTitle (error_dialog -> shellwidget, "Apply Load");
    if (active_distributed_number == -1) {
	error ("No active load defined.");
	return;
    }

    SetEditMode ( );
    ChangeStatusLine ("Select element:", True);

    XtRemoveAllCallbacks (drawing, XtNbuttonCallback);
    XtAddCallback (drawing, XtNbuttonCallback, ApplyLoadCB, NULL);

    XtOverrideTranslations (entry,
	XtParseTranslationTable ("<Key>Return: ApplyLoadAP()"));
}
