/*
    This file is part of the FElt finite element analysis package.
    Copyright (C) 1993 Jason I. Gobat and Darren C. Atkinson

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

# include <stdio.h>
# include <string.h>
# include <stdlib.h>
# include <X11/Intrinsic.h>
# include "fe.h"
# include "forms.h"
# include "dialog.h"
# include "problem.h"
# include "globals.h"
# include "Drawing.h"
# include "vfe.h"
# include "util.h"
# include "objects.h"
# include "procedures.h"
# include "error.h"

static int count;


static int SetForceNames (item)
    Item item;
{
    if (active_force == (Force) item)
	active_force_number = count;

    force_names [count ++] = ((Force) item) -> name;
    return 0;
}


void GatherForceNames ( )
{
    int size;


    if ((size = TreeSize (problem.force_tree)) <= 0)
	return;

    count = 0;
    size *= sizeof (String);
    force_names = (String *) XtRealloc ((char *) force_names, size);
    (void) TreeSetIterator (problem.force_tree, SetForceNames);
    (void) TreeIterate (problem.force_tree);
}


void GetForceDefinition ()
{
   String 			result[7];
   extern EntryRecord		force_entries;
   unsigned			i, status;
   Force			force;
   Item			        found;
 
   SetShellTitle (force_form -> shellwidget, "Define Force");
   status = GetEntryFormValues (force_form, force_entries, result);
   if (status) {
      if (strlen (result[0]) > 0)  {
	 force = CreateForce (XtNewString (result [0]));
	 for (i = 1; i <= 6; i ++)
	    force -> force [i] = atof (result [i]);

	 found = TreeInsert (problem.force_tree, (Item) force);
	 if (found != (Item) force) {
	    error ("Force %s already exists.", force -> name);
	    DestroyForce (force);
	    return;
	 }

	 active_force = force;
	 GatherForceNames ( );
	 changeflag = True;

      } else
	 error ("No force name specified.");
   }
}

void SetActiveForce ()
{
   int             	result, size;
   struct force		dummy;

   SetShellTitle (force_list -> shellwidget, "Set Active Force");
   if ((size = TreeSize (problem.force_tree)) <= 0) {
      error ("No forces defined.");
      return;
   }

   SetListItems (force_list, force_names, size);
   result = GetListFormValue (force_list, active_force_number);

   if (result > -1) {
      dummy.name = force_names [result];
      active_force = (Force) TreeSearch (problem.force_tree, &dummy);
      active_force_number = result;
      changeflag = True;
   }
}


void EditForceInfo ()
{
    int			i, status, size;
    String		result [7];
    extern EntryRecord	force_entries;
    Force		force;
    static String	suggestion [7];
    static int		flag = 0;
    struct force	dummy;
    Item		found;


    if (!flag) {
	flag = 1;
	for (i = 0; i < 7; i ++)
	    suggestion [i] = (String) XtMalloc (sizeof (char) * 80);
    }

    SetShellTitle (force_list -> shellwidget, "Edit Force");
    if ((size = TreeSize (problem.force_tree)) <= 0) {
	error ("No forces defined.");
	return;
    }

    SetListItems (force_list, force_names, size);
    status = GetListFormValue (force_list, active_force_number);

    if (status == -1)
	return;

    dummy.name = force_names [status];
    force = (Force) TreeSearch (problem.force_tree, &dummy);

    strcpy (suggestion [0], force -> name);
    for (i = 1; i <= 6; i ++)
	sprintf (suggestion [i], "%g", force -> force [i]);

    FillTextBuffers (force_form, force_entries, suggestion);

    SetShellTitle (force_form -> shellwidget, "Edit Force");
    status = GetEntryFormValues (force_form, force_entries, result);

    if (status) {
	if (strlen (result [0]) > 0) {
	    if (strcmp (result [0], force -> name))
		force = CreateForce (XtNewString (result [0]));

	    for (i = 1; i <= 6; i ++)
		force -> force [i] = atof (result [i]);

	    found = TreeInsert (problem.force_tree, (Item) force);
	    if (found != (Item) force) {
		error ("Force %s already exists.", force -> name);
		DestroyForce (force);
		return;
	    }

	    active_force = force;
	    GatherForceNames ( );
	    changeflag = True;
	} else
	    error ("No force name specified.");
    }
}


static Force force;
static Boolean first_time;

static int ForceRef (item)
    Item item;
{
    int status;

    if (((Node) item) -> force == force) {
	if (first_time == True) {
	    first_time = False;
	    SetShellTitle (proceed_dialog -> shellwidget, "Delete Force");
	    status = PopupDialog (proceed_dialog, "Force is still in use.  Proceed?", NULL, NULL);
	    if (status != Okay)
		return 1;
	}
	((Node) item) -> force = NULL;
	if (displayed_node == (Node) item && XtIsRealized (nodeShell) == True)
	    GetNodeInformation (displayed_node);
    }

    return 0;
}


void DeleteForce ()
{
    int			result, size;
    struct force	dummy;


    SetShellTitle (force_list -> shellwidget, "Delete Force");
    if ((size = TreeSize (problem.force_tree)) <= 0) {
	error ("No forces defined.");
	return;
    }

    SetListItems (force_list, force_names, size);
    result = GetListFormValue (force_list, -1);

    if (result > -1) {
	first_time = True;
	dummy.name = force_names [result];
	force = (Force) TreeSearch (problem.force_tree, &dummy);
	(void) TreeSetIterator (problem.node_tree, ForceRef);
	if (TreeIterate (problem.node_tree))
	    return;

	DestroyForce ((Force) TreeDelete (problem.force_tree, &dummy));
	if (active_force_number == result)
	    active_force_number = -1;

	GatherForceNames ( );
	changeflag = True;
    }
}


void DoApplyForce (node)
    Node node;
{
    static char message [40];


    node -> force = active_force;

    if (XtIsRealized (nodeShell) == True)
	GetNodeInformation (node);

    sprintf (message, "Applied to node %d.  Select node:", node -> number);
    ChangeStatusLine (message, True);
    changeflag = True;
}


void ApplyForceCB (w, client_data, call_data)
    Widget    w;
    XtPointer client_data;
    XtPointer call_data;
{
    DrawingReport   *report;
    FigureAttributes attributes;
    Figure           figure;
    Node             node;
    Drawn            drawn;


    report = (DrawingReport *) call_data;

    if (report -> event -> type != ButtonPress)
	return;

    if (report -> event -> xbutton.button == 3)
	QuitEdit ( );

    if (report -> event -> xbutton.button != 1)
	return;

    figure = DW_FindFigure (w, report -> unsnapped.x, report -> unsnapped.y);

    if (figure == NULL)
	return;

    DW_GetAttributes (w, figure, &attributes);
    if (attributes.user_data == NULL)
	return;

    node = (Node) attributes.user_data;
    drawn = (Drawn) node -> aux;
    if (drawn -> type != DrawnNode)
	return;

    DoApplyForce (node);
}


void ApplyForceAP ( )
{
    char       *status;
    struct node dummy;
    Item        found;


    if ((status = GetTextNumber (&dummy.number)) != NULL)
	return;

    found = TreeSearch (problem.node_tree, (Item) &dummy);
    if (found == NULL) {
	error ("Node %d does not exist.", dummy.number);
	return;
    }

    DoApplyForce (found);
}


void EditApplyForce ( )
{
    SetShellTitle (error_dialog -> shellwidget, "Apply Force");
    if (active_force_number == -1) {
	error ("No active force defined.");
	return;
    }

    SetEditMode ( );
    ChangeStatusLine ("Select node:", True);

    XtRemoveAllCallbacks (drawing, XtNbuttonCallback);
    XtAddCallback (drawing, XtNbuttonCallback, ApplyForceCB, NULL);

    XtOverrideTranslations (entry,
	XtParseTranslationTable ("<Key>Return: ApplyForceAP()"));
}
