/*
    This file is part of the FElt finite element analysis package.
    Copyright (C) 1993 Jason I. Gobat and Darren C. Atkinson

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

# include <stdio.h>
# include <X11/Intrinsic.h>
# include <X11/StringDefs.h>
# include <X11/Xaw/Viewport.h>
# include "fe.h"
# include "dialog.h"
# include "forms.h"
# include "problem.h"
# include "globals.h"
# include "procedures.h"
# include "objects.h"
# define ELEMENTS
# include "element.h"
# include "Drawing.h"
# include "vfe.h"


void OpenFile ()
{
   String	ans;
   unsigned	selected;
   static char	suggestion [256] = "";

   if (changeflag == True)
      if (!QuerySave ()) return;
         
   SetShellTitle (file_dialog -> shellwidget, "Open Problem");
   selected = PopupDialog (file_dialog, "File to load:", suggestion, &ans);
   strcpy (suggestion, ans);

   if (selected == Okay && ans [0]) {
      changeflag = False;
      strcpy (filename, ans);
      if (!ReadFeltFile ()) {
	 UpdateTitle ();
         ReadVelvetFile ();
      }
   } 
}

void RestoreOriginal ()
{
   unsigned	selected;

   SetShellTitle (proceed_dialog -> shellwidget, "Restore Problem");

   if (changeflag == True) {
      selected = PopupDialog (proceed_dialog, "Are you sure?", NULL, NULL);
 
      if (selected != Okay)
	 return;

      changeflag = False;
  }

  if (!ReadFeltFile ( )) {
      UpdateTitle ( );
      ReadVelvetFile ( );
  }
} 
   
void WriteNamedFile ()
{
   String	ans;
   unsigned	selected;

   SetShellTitle (file_dialog -> shellwidget, "Save Problem");
   selected = PopupDialog (file_dialog, "Save as:", "", &ans);
   if (selected == Okay && ans [0]) {
      strcpy (filename, ans);
      UpdateTitle ();

      if (!WriteVFeltFile ())
         WriteVelvetFile ();

      changeflag = False;
   }
}


int WriteVFeltFile ()
{
    FILE *output;


    if ((output = fopen (filename, "w")) == NULL) {
	error ("Could not create file %s.", filename);
	return 1;
    }

    CompactOperation (output);
    (void) fclose (output);

    return 0;
}


int ReadFeltFile ( )
{
    int   status, i;
    FILE *input;
    int   usecpp = 1;
    char *cpp = "/lib/cpp";
    Element elt;

    if (freopen (filename, "r", stdin) == NULL) {
	error ("Could not access file %s.", filename);
	return 1;
    }

    if (usecpp) {
	if ((input = popen (cpp, "r")) == NULL) {
	    error ("Could not execute %s.", cpp);
	    return 1;
	}
    } else
	input = stdin;

    DW_RemoveAll (drawing);
    BufferErrors (True);
    status = GetProblem (input, &element, &node, &numelts, &numnodes, &title);
    (void) pclose (input);
    BufferErrors (False);

    if (status) {
	StartNew ( );
	return 1;
    }

    DrawProblem (0.0);

    active_constraint = (Constraint) TreeMinimum (problem.constraint_tree);
    active_force = (Force) TreeMinimum (problem.force_tree);
    active_material = (Material) TreeMinimum (problem.material_tree);
    active_distributed = (Distributed) TreeMinimum (problem.distributed_tree);
    elt = (Element) TreeMinimum (problem.element_tree);
    if (elt != NULL) {
	active_definition = elt -> definition;
	for (i = 0; i < XtNumber (ElementArray); i ++)
	    if (active_definition == ElementArray [i].definition)
		active_definition_number = i;
    } else {
	active_definition = NULL;
	active_definition_number = -1;
    }

    GatherDistributedNames ( );
    GatherConstraintNames ( );
    GatherMaterialNames ( );
    GatherForceNames ( );
    return 0;
}

static FILE	*vfile;

static int FigureWriteIterator (item)
   Item		item;
{
   FigureAttributes	attr;
   Figure		figure;
   float		fdata [20];
   int			length;
   unsigned		j;

   figure = (Figure) item;

   DW_GetAttributes (drawing, figure, &attr);
   fwrite ((FigureType *) &(attr.type), sizeof(FigureType), 1, vfile);
   
   switch (attr.type) {

   case LineFigure:
      fdata [0] = attr.points [0].x;
      fdata [1] = attr.points [0].y;
      fdata [2] = attr.points [1].x;
      fdata [3] = attr.points [1].y;
      fwrite ((float *) fdata, sizeof(float), 4, vfile);
      break;

   case RectangleFigure:
      fdata [0] = attr.x;
      fdata [1] = attr.y;
      fdata [2] = attr.width;
      fdata [3] = attr.height;
      fwrite ((float *) fdata, sizeof(float), 4, vfile);
      break;

   case TextFigure:
      fdata [0] = attr.x;
      fdata [1] = attr.y;
      length = strlen(attr.text);
      fwrite ((float *) fdata, sizeof(float), 2, vfile);
      fwrite ((int *) &length, sizeof(int), 1, vfile);
      fwrite ((char *) attr.text, sizeof(char), length, vfile);
      break;

   case ArcFigure:
      fdata [0] = attr.x;
      fdata [1] = attr.y;
      fdata [2] = attr.width;
      fdata [3] = attr.height;
      fdata [4] = attr.arc_start;
      fdata [5] = attr.arc_length;
      fwrite ((float *) fdata, sizeof(float), 6, vfile);
      break;

   case PolygonFigure:
      fwrite ((int *) (&attr.npoints), sizeof(int), 1, vfile);
      for (j = 0 ; j < attr.npoints ; j++)
         fwrite ((Point *) (&attr.points[j]), sizeof(Point), 1, vfile); 
      break;

   case GroupFigure:
      break;
   }   

   return 0;
}

String FormVelvetFilename ()
{
   static char	name [2048];
   int		length;

   length = strlen (filename);
   if (length >= 4) {
      if (strcmp (&(filename[length-4]), ".flt") == 0) {
         strcpy (name, filename);
         strcpy (name + length - 4, ".vlv");
      } else {
         strcpy (name, filename);
         strcat (name, ".vlv");  
      }
   }
   else {
      strcpy (name, filename);
      strcat (name, ".vlv");  
   }
   return name;
}
   
void WriteVelvetFile ()
{
   Arg			arglist [3];
   Cardinal		count;
   Position		x_pos,y_pos;
   Dimension		width, height;
   float		scale;
   int			idata [7];
   float		fdata [7];
   int			num_figs;

   SetShellTitle (error_dialog -> shellwidget, "Write Velvet File");
   vfile = fopen (FormVelvetFilename(), "w");

   if (vfile == NULL) {
      error ("Could not write settings file.");
      return;
   }

   count = 0;
   XtSetArg (arglist [count], XtNx, &x_pos); count++;
   XtSetArg (arglist [count], XtNy, &y_pos); count++;
   XtSetArg (arglist [count], XtNxScale, &scale); count++;
   XtGetValues (drawing, arglist, count);

   count = 0;
   XtSetArg (arglist [count], XtNwidth, &width); count++;
   XtSetArg (arglist [count], XtNheight, &height); count++;
   XtGetValues (viewport, arglist, count);

   idata[0] = number_status;
   idata[1] = snap_status;
   idata[2] = grid_status;
   idata[3] = x_pos;
   idata[4] = y_pos;
   idata[5] = height;
   idata[6] = width;

   fwrite ((int *) idata, sizeof (int), 7, vfile); 

   fdata[0] = xmin;
   fdata[1] = xmax;
   fdata[2] = ymin;
   fdata[3] = ymax;
   fdata[4] = scale;
   fdata[5] = grid_size;
   fdata[6] = snap_size;

   fwrite ((float *) fdata, sizeof (float), 7, vfile);
 
	/*
	 * write out all the tools, switching on type so we know which	
	 * things to write ... we'll just do the reverse on the way back in
	 */

   num_figs = TreeSize (figure_tree);
   fwrite ((int *) &num_figs, sizeof (int), 1, vfile);

   TreeSetIterator (figure_tree, FigureWriteIterator);
   TreeIterate (figure_tree);

   fclose (vfile); 
}

void ReadVelvetFile ()
{
   Arg		arglist [10];
   Cardinal	count;
   Position	x_pos,y_pos;
   float	scale;
   int		idata [7];
   float	fdata [7];
   Dimension	width,height;
   Boolean	number_state,
		grid_state,
		snap_state;
   int		num_figs;
   int		length;
   char		text [256];
   unsigned	i,j;
   FigureType	type;
   Figure	figure;
   int		npoints;
   Point	point [100];

   SetShellTitle (error_dialog -> shellwidget, "Read Velvet File");
   vfile = fopen (FormVelvetFilename(), "r");

   if (vfile == NULL) 
      return;

   fread ((int *) idata, sizeof (int), 7, vfile); 

   number_state = idata[0];
   snap_state = idata[1];
   grid_state = idata[2];

   if (number_state != number_status) 
      ToggleNumberStatus ();
   if (grid_state != grid_status) 
      ToggleGridStatus ();
   if (snap_state != snap_status)
      ToggleSnapStatus ();
   
   x_pos = idata[3];
   y_pos = idata[4];
   height = idata[5];
   width = idata[6];

   fread ((float *) fdata, sizeof (float), 7, vfile);

   xmin = fdata[0];
   xmax = fdata[1];
   ymin = fdata[2];
   ymax = fdata[3];
   scale = fdata[4];
   grid_size = fdata[5];
   snap_size = fdata[6];

	/*
	 * read in all the tools, switching on type so we know which	
	 * things to read ... 
	 */

   if (DW_SetForeground (drawing, tool_color) == False)
	(void) DW_SetForeground (drawing, "black");

   if (DW_SetFont (drawing, text_font) == False)
	(void) DW_SetFont (drawing, "fixed");

   fread ((int *) &num_figs, sizeof(int), 1, vfile);
   for (i = 0 ; i < num_figs ; i++) {
      fread ((FigureType *) &type, sizeof(FigureType), 1, vfile);
      switch (type) {

      case LineFigure:
         fread ((float *) fdata, sizeof(float), 4, vfile);
         figure = DW_DrawLine (drawing, fdata [0],fdata [1],
                               fdata [2],fdata [3]);
         break;

      case RectangleFigure:
         fread ((float *) fdata, sizeof(float), 4, vfile);
         figure = DW_DrawRectangle (drawing, True, fdata [0], fdata [1], 
                                    fdata [2], fdata [3]);
         break;

      case TextFigure:
         fread ((float *) fdata, sizeof(float), 2, vfile);
         fread ((int *) &length, sizeof(int), 1, vfile);
         fread ((char *) text, sizeof(char), length, vfile);  
         figure = DW_DrawText (drawing, True, fdata[0], fdata[1], text);
         break;

      case ArcFigure:
         fread ((float *) fdata, sizeof(float), 6, vfile);
         figure = DW_DrawArc (drawing, True, fdata [0], fdata [1],
                              fdata [2], fdata [3], fdata [4], fdata [5]);
         break;

      case PolygonFigure:
         fread ((int *) &npoints, sizeof(int), 1, vfile);
         for (j = 0 ; j < npoints ; j++) 
            fread ((Point *) &(point[j]), sizeof(Point), 1, vfile);
 
         figure = DW_DrawPolygon (drawing, True, point, npoints);
         break;

      case GroupFigure:
	 break;
      }
      TreeInsert (figure_tree, figure);
   }

   fclose (vfile); 

   count = 0;
   XtSetArg (arglist [count], XtNgridSize, Float2Arg (grid_size)); count++;
   XtSetArg (arglist [count], XtNsnapSize, Float2Arg (snap_size)); count++;
   XtSetArg (arglist [count], XtNxMin, Float2Arg (xmin)); count++;
   XtSetArg (arglist [count], XtNxMax, Float2Arg (xmax)); count++;
   XtSetArg (arglist [count], XtNyMin, Float2Arg (ymin)); count++;
   XtSetArg (arglist [count], XtNyMax, Float2Arg (ymax)); count++;
   XtSetArg (arglist [count], XtNxScale, Float2Arg(scale)); count++;
   XtSetArg (arglist [count], XtNyScale, Float2Arg(scale)); count++;
   XtSetValues (drawing, arglist, count);

   count = 0;
   XtSetArg (arglist [count], XtNheight, height); count++;
   XtSetArg (arglist [count], XtNwidth, width); count++;
   XtSetValues (viewport, arglist, count);

   XawViewportSetCoordinates (viewport, -x_pos, -y_pos);
}


void StartNew ()
{
   if (changeflag) 
      if (!QuerySave ()) return; 

   changeflag = False;
   filename [0] = 0;
   UpdateTitle ();
   title = XtNewString ("Unnamed");
   DW_RemoveAll (drawing);
   DestroyProblem ((ItemDestructor) DestroyMaterial);
   TreeDestroy (figure_tree);

   active_constraint = NULL;
   active_constraint_number = -1;
   active_force = NULL;
   active_force_number = -1;
   active_distributed = NULL;
   active_distributed_number = -1;
   active_material = NULL;
   active_material_number = -1;
   active_definition = NULL;
   active_definition_number = -1;

   problem.node_tree        = TreeCreate (node_cmp);
   problem.element_tree     = TreeCreate (element_cmp);
   problem.force_tree       = TreeCreate (force_cmp);
   problem.material_tree    = TreeCreate (material_cmp);
   problem.constraint_tree  = TreeCreate (constraint_cmp);
   problem.distributed_tree = TreeCreate (distributed_cmp);
   figure_tree		    = TreeCreate (figure_cmp);
}

Boolean QuerySave ()
{
    unsigned	selected;

    SetShellTitle (qsave_dialog -> shellwidget, "Are You Sure?");
    selected = PopupDialog (qsave_dialog, "Changes not saved, save now?", 
                  NULL, NULL);

    if (selected == Yes) {
       if (!WriteVFeltFile ())
          WriteVelvetFile ();

       return True;
    }
    else if (selected == Cancel) return False;
  
    return True;
}
    
void QuitVelvet ()
{
   if (changeflag) 
      if (!QuerySave ()) return;

   XtUnmapWidget (toplevel);
   XtDestroyApplicationContext (app_context);
   exit (0);
}
