/*
    This file is part of the FElt finite element analysis package.
    Copyright (C) 1993 Jason I. Gobat and Darren C. Atkinson

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/************************************************************************
 * File:	error.c							*
 *									*
 * Description:	This file contains the error handling routines.		*
 *									*
 * History:	V1.0 by Jason Gobat and Darren C. Atkinson		*
 ************************************************************************/

# include <stdio.h>
# include <unistd.h>
# include <varargs.h>
# include <X11/Intrinsic.h>
# include "fe.h"
# include "error.h"
# include "forms.h"
# include "dialog.h"
# include "Tree.h"
# include "globals.h"
# include "procedures.h"


static Boolean buffer_errors = False;
static Boolean errors = False;
static FILE *output;


/************************************************************************
 * Function:	 error							*
 *									*
 * Parameters:	 error message						*
 *									*
 * Return value: none							*
 *									*
 * Calls:	 none							*
 *									*
 * Called by:	 other routines						*
 *									*
 * Global data:	 none							*
 *									*
 * Description:	 Error prints the error message specified as a format	*
 *		 string and arguments to the error dialog.		*
 ************************************************************************/

/* VARARGS */
void error (va_alist)
    va_dcl
{
    char   *format;
    va_list ap;
    char    error_msg [256];


    if (buffer_errors == True) {
	va_start (ap);
	format = va_arg (ap, char *);
	vfprintf (output, format, ap);
	fprintf (output, "\n");
	va_end (ap);
	errors = True;
    } else {
	va_start (ap);
	format = va_arg (ap, char *);
	vsprintf (error_msg, format, ap);
	va_end (ap);

	PopupDialog (error_dialog, error_msg, NULL, NULL);
    }
}


/************************************************************************
 * Function:	 Fatal							*
 *									*
 * Parameters:	 error message						*
 *									*
 * Return value: none							*
 *									*
 * Calls:	 none							*
 *									*
 * Called by:	 other routines						*
 *									*
 * Global data:	 none							*
 *									*
 * Description:	 Fatal prints the error message specified as a format	*
 *		 string and arguments to standard error and exits the	*
 *		 program.						*
 ************************************************************************/

/* VARARGS */
void Fatal (va_alist)
    va_dcl
{
    char   *format;
    va_list ap;

    va_start (ap);
    format = va_arg (ap, char *);
    fprintf (stderr, "velvet: ");
    vfprintf (stderr, format, ap);
    fprintf (stderr, "\n");
    va_end (ap);
    exit (1);
}


void BufferErrors (flag)
    Boolean flag;
{
    static char *name;

    if (flag == True && buffer_errors == False) {
	name = tempnam (NULL, "velv");
	if ((output = fopen (name, "w")) == NULL) {
	    error ("Could not create temporary file for output.");
	    return;
	}
	buffer_errors = flag;
	errors = False;

    } else if (flag == False && buffer_errors == True) {
	(void) fclose (output);
	if (errors == True) {
	    SetShellTitle (outputShell, "Error messages");
	    UpdateOutputWindow (name);
	}
	(void) unlink (name);
	XtFree ((char *) name);
	buffer_errors = flag;
    }
}
