/*
    This file is part of the FElt finite element analysis package.
    Copyright (C) 1993 Jason I. Gobat and Darren C. Atkinson

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

# include <stdio.h>
# include <string.h>
# include <stdlib.h>
# include <X11/Intrinsic.h>
# include <X11/StringDefs.h>
# include <X11/Xaw/Simple.h>
# include "fe.h"
# include "forms.h"
# include "dialog.h"
# include "Drawing.h"
# include "procedures.h"
# define ELEMENTS
# include "element.h"
# include "problem.h"
# include "globals.h"
# include "vfe.h"
# include "util.h"
# include "objects.h"

void GetElementInformation (element)
   Element element;
{
   Arg		arglist [2];
   static char	information [1024];
   unsigned	i; 

   displayed_element = element;

   sprintf (information,"Number:   %d\n\
Type:     %s\n\
Nodes: ",element -> number, ElementArray[element -> definition -> type].name);

   for (i = 1 ; i <= element -> definition -> numnodes ; i++) 
      sprintf (information + strlen(information),"   %d", 
               element -> node [i] ? element -> node[i] -> number : 0);

   sprintf (information + strlen(information),"\nMaterial: %s\nLoads:",
            element -> material -> name);

   if (element -> numdistributed == 0)
      sprintf (information + strlen (information),"    (none)");
   else
      for (i = 1 ; i <= element -> numdistributed ; i++)
         sprintf (information + strlen (information),"    %s",
                  element -> distributed[i] -> name);
   
   sprintf (information + strlen (information),"\nStresses: ");
   if (element -> numstresses == 0 || element -> stress == NULL)
      sprintf (information + strlen (information),"(none)");
   else {
      for (i = 1; i <= element -> numstresses ; i++) {
          sprintf (information + strlen (information)," % 11.5g", 
                   element -> stress[i]);
          if (i % 3 == 0 && i != element -> numstresses) 
             sprintf (information + strlen (information),"\n          ");
      } 
   }

   XtSetArg (arglist [0], XtNstring, information);
   XtSetArg (arglist [1], XtNheight, (Dimension) (95 + element->numstresses/3));
   XtSetValues (elementwin, arglist, 2);

   XtPopup (elementShell, XtGrabNone);
}


void SetActiveElementType ()
{
   int             	result;
   String	        *elements;	 
   static unsigned	flag = 1;
   unsigned		i;

   if (flag) {
      elements = (String *) XtMalloc (sizeof (String)*XtNumber (ElementArray));

      for (i = 0 ; i < XtNumber (ElementArray) ; i++) 
         elements [i] = ElementArray [i].name;

      SetListItems (element_list, elements, XtNumber (ElementArray)-1);

      flag = 0;
   }

   SetShellTitle (element_list -> shellwidget, "Set Active Element");
   result = GetListFormValue (element_list, active_definition_number);
   
   if (result > -1) {
	active_definition_number = result;
	active_definition = ElementArray [result].definition;
	changeflag = True;
   }
}


void EditElementInfo (element)
    Element element;
{
    int			i, status;
    String		result [7];
    extern EntryRecord	element_entries;
    static String	suggestion [7];
    static int		flag = 0;
    struct distributed	distributed;
    struct material	material;
    struct element	dummy;
    struct node		node;
    unsigned		numloads;
    unsigned		num;
    Boolean		newnodes;
    Distributed		newloads [4];
    Item		found;
    String		start;
    String		end;
    int			vals [MaxNodesPerElement];
    int			numvals;
    unsigned		shapenodes;
    Node		ref;
    Drawn		drawn;
    Figure		fig;
    FigureAttributes	attributes;


    if (!flag) {
	flag = 1;
	for (i = 0; i < 7; i ++)
	    suggestion [i] = (String) XtMalloc (sizeof (char) * 80);
    }

    sprintf (suggestion [0], "%d", element -> number);

    suggestion [2] [0] = 0;
    for (i = 1; i <= element -> definition -> numnodes; i ++)
	sprintf (suggestion [2] + strlen (suggestion [2]), "%d  ",
		 element -> node ? element -> node [i] -> number: 0);

    strcpy (suggestion [1], ElementArray [element -> definition -> type].name);
    strcpy (suggestion [3], element -> material ? element -> material -> name : "");
    strcpy (suggestion [4], element -> numdistributed >= 1 ?
			    element -> distributed [1] -> name : "");
    strcpy (suggestion [5], element -> numdistributed >= 2 ?
			    element -> distributed [2] -> name : "");
    strcpy (suggestion [6], element -> numdistributed >= 3 ?
			    element -> distributed [3] -> name : "");

    FillTextBuffers (element_form, element_entries, suggestion);
    SetShellTitle (element_form -> shellwidget, "Edit Element");
    status = GetEntryFormValues (element_form, element_entries, result);

    if (status) {
	if (strlen (result [0]) > 0) {
	    dummy.number = atoi (result [0]);
	    if (element -> number != dummy.number) {
		found = TreeSearch (problem.element_tree, &dummy);
		if (found == NULL) {
		    error ("Element %d does not exist.", dummy.number);
		    return;
		}

		element = (Element) found;
	    }

	    numvals = 0;
	    start = result [2];

	    while (*start) {
		vals [numvals ++] = strtol (start, &end, 10);
		if (*end != ' ' && *end) {
		    error ("Illegal node specified.");
		    return;
		}
		if (start == end) {
		    numvals --;
		    break;
		}
		start = end;
	    }

	    if (numvals != element -> definition -> numnodes) {
		error ("Incorrect number of nodes specified.");
		return;
	    }

	    newnodes = False;
	    shapenodes = element -> definition -> shapenodes;

	    for (i = 0; i < numvals; i ++) {
		node.number = vals [i];
		if (node.number == 0) {
		    if (i <= shapenodes) {
			error ("Illegal zero node.\n");
			return;
		    }

		} else {
		    found = TreeSearch (problem.node_tree, &node);
		    if (found == NULL) {
			error ("Node %d does not exist.", node.number);
			return;
		    }
		}

		num = element -> node [i + 1] ?
				element -> node [i + 1] -> number : 0;

		if (node.number != num)
		    newnodes = True;
	    }

	    if (strlen (result [3]) == 0) {
		error ("No material specified.");
		return;
	    } else {
		material.name = result [3];
		found = TreeSearch (problem.material_tree, &material);
		if (found == NULL) {
		    error ("Material %s does not exist.", material.name);
		    return;
		}

		element -> material = (Material) found;
	    }

	    if (strlen (result [4]) == 0) {
		if (strlen (result [5]) || strlen (result [6])) {
		    error ("First load is absent.");
		    return;
		}
		numloads = 0;
	    } else {
		distributed.name = result [4];
		found = TreeSearch (problem.distributed_tree, &distributed);
		if (found == NULL) {
		    error ("Load %s does not exist.", distributed.name);
		    return;
		}

		newloads [1] = (Distributed) found;
		numloads = 1;
	    }

	    if (strlen (result [5]) == 0) {
		if (strlen (result [6])) {
		    error ("Second load is absent.");
		    return;
		}
	    } else {
		distributed.name = result [5];
		found = TreeSearch (problem.distributed_tree, &distributed);
		if (found == NULL) {
		    error ("Load %s does not exist.", distributed.name);
		    return;
		}

		newloads [2] = (Distributed) found;
		numloads = 2;
	    }

	    if (strlen (result [6]) > 0) {
		distributed.name = result [6];
		found = TreeSearch (problem.distributed_tree, &distributed);
		if (found == NULL) {
		    error ("Load %s does not exist.", distributed.name);
		    return;
		}

		newloads [3] = (Distributed) found;
		numloads = 3;
	    }

	    element -> numdistributed = numloads;
	    for (i = 1; i <= numloads; i ++)
		element -> distributed [i] = newloads [i];

	    if (newnodes == True) {
		fig = ((Drawn) (element -> aux)) -> figure;
		if (fig != NULL)
		    DW_GetAttributes (drawing, fig, &attributes);

		for (i = 0; i < numvals; i ++) {
		    node.number = vals [i];
		    ref = (Node) TreeSearch (problem.node_tree, &node);
		    if (element -> node [i + 1] != NULL) {
			drawn = (Drawn) element -> node [i + 1] -> aux;
			drawn -> ref_count --;
		    }

		    element -> node [i + 1] = ref;
		    if (ref != NULL)
			((Drawn) (ref -> aux)) -> ref_count ++;

		    if (i < shapenodes && fig != NULL) {
			attributes.points [i].x = ref -> x;
			attributes.points [i].y = ref -> y;
			if (i == 0) {
			    attributes.points [shapenodes].x = ref -> x;
			    attributes.points [shapenodes].y = ref -> y;
			}
		    }
		}

		DW_SetAttributes (drawing, fig, DW_FigurePoints, &attributes);
	    }

	    if (XtIsRealized (elementShell))
		GetElementInformation (element);

	    changeflag = True;
	} else
	    error ("No element number specified.");
    }
}


void DeleteElementGroup (figures, nfigures)
    Figure  *figures;
    unsigned nfigures;
{
    unsigned         i;
    unsigned         j;
    Figure           fig;
    Drawn            drawn;
    Element          element;
    Boolean          firsttime;
    Boolean	     newinfo;
    FigureAttributes attr;


    firsttime = True;
    newinfo = False;

    for (i = 0; i < nfigures; i ++) {
	fig = figures [i];
	DW_GetAttributes (drawing, fig, &attr);

	if (attr.user_data == NULL || attr.type == TextFigure)
	    continue;

	element = (Element) attr.user_data;
	drawn = (Drawn) element -> aux;
	if (drawn -> type != DrawnElement)
	    continue;

	if (firsttime == True) {
	    firsttime = False;
	    DW_SetAutoRedraw (drawing, False);
	}

	for (j = 1; j <= element -> definition -> numnodes; j ++)
	    if (element -> node [j] != NULL)
		((Drawn) element -> node [j] -> aux) -> ref_count --;

	if (element == displayed_element)
	    newinfo = True;

	DW_RemoveFigure (drawing, drawn -> figure);
	DW_RemoveFigure (drawing, drawn -> label);
	(void) TreeDelete (problem.element_tree, element);
	DestroyElement (element);
    }


    if (newinfo == True && XtIsRealized (elementShell)) {
	element = TreeMinimum (problem.element_tree);
	if (element != NULL)
	    GetElementInformation (element);
	else {
	    XtPopdown (elementShell);
	    XtUnrealizeWidget (elementShell);
	}
    }

    if (firsttime == False) {
	DW_SetAutoRedraw (drawing, True);
	changeflag = True;
    }

    XtFree ((char *) figures);
}


void DoDeleteElt (element)
    Element element;
{
    unsigned i;
    Element newelement;
    static char message [80];
    Drawn drawn = (Drawn) element -> aux;


    if (drawn -> figure != NULL) {
	DW_SetAutoRedraw (drawing, False);
	DW_RemoveFigure  (drawing, drawn -> figure);
	DW_SetAutoRedraw (drawing, True);
    }

    for (i = 1; i <= element -> definition -> numnodes; i ++)
	if (element -> node [i] != NULL)
	    ((Drawn) (element -> node [i] -> aux)) -> ref_count --;

    sprintf (message, "Element %d deleted.  Select element:", element -> number);

    if (element == displayed_element && XtIsRealized (elementShell)) {
	newelement = TreePredecessor (problem.element_tree, element);
	if (newelement == NULL)
	    newelement = TreeSuccessor (problem.element_tree, element);
	if (newelement == NULL) {
	    XtPopdown (elementShell);
	    XtUnrealizeWidget (elementShell);
	} else
	    GetElementInformation (newelement);
    }

    (void) TreeDelete (problem.element_tree, element);

    DestroyElement (element);
    ChangeStatusLine (message, True);
    changeflag = True;
}


void DeleteEltCB (w, client_data, call_data)
    Widget    w;
    XtPointer client_data;
    XtPointer call_data;
{
    DrawingReport   *report;
    FigureAttributes attributes;
    Figure           figure;
    Element          element;
    Drawn            drawn;


    report = (DrawingReport *) call_data;

    if (report -> event -> type != ButtonPress)
	return;

    if (report -> event -> xbutton.button == 3)
	QuitEdit ( );

    if (report -> event -> xbutton.button == 2) {
	DeleteGroup (call_data, DeleteElementGroup);
	return;
    }

    if (report -> event -> xbutton.button != 1)
	return;

    figure = DW_FindFigure (w, report -> unsnapped.x, report -> unsnapped.y);

    if (figure == NULL)
	return;

    DW_GetAttributes (w, figure, &attributes);
    if (attributes.user_data == NULL)
	return;

    element = (Element) attributes.user_data;
    drawn = (Drawn) element -> aux;
    if (drawn -> type != DrawnElement)
	return;

    DoDeleteElt (element);
}


void DeleteEltAP ( )
{
    char          *status;
    struct element dummy;
    Item           found;


    if ((status = GetTextNumber (&dummy.number)) != NULL) {
	if (!strcmp (status, "w"))
	    DeleteGroup (NULL, DeleteElementGroup);
	return;
    }

    found = TreeSearch (problem.element_tree, (Item) &dummy);
    if (found == NULL) {
	error ("Element %d does not exist.", dummy.number);
	return;
    }

    DoDeleteElt (found);
}


void EditDeleteElement ( )
{
    Arg		arglist [1];

    SetEditMode ( );
    SetShellTitle (error_dialog -> shellwidget, "Delete Element");
    ChangeStatusLine ("Select element:", True);

    XtSetArg (arglist [0], XtNcursorName, "dotbox");
    XtSetValues (drawing, arglist, 1);

    XtRemoveAllCallbacks (drawing, XtNbuttonCallback);
    XtAddCallback (drawing, XtNbuttonCallback, DeleteEltCB, NULL);

    XtOverrideTranslations (entry,
	XtParseTranslationTable ("<Key>Return: DeleteEltAP()"));
}


void EditElementCB (w, client_data, call_data)
    Widget    w;
    XtPointer client_data;
    XtPointer call_data;
{
    DrawingReport   *report;
    FigureAttributes attributes;
    Figure           figure;
    Element          element;
    Drawn            drawn;


    report = (DrawingReport *) call_data;

    if (report -> event -> type != ButtonPress)
	return;

    if (report -> event -> xbutton.button == 3)
	QuitEdit ( );

    if (report -> event -> xbutton.button != 1)
	return;

    figure = DW_FindFigure (w, report -> unsnapped.x, report -> unsnapped.y);

    if (figure == NULL)
	return;

    DW_GetAttributes (w, figure, &attributes);
    if (attributes.user_data == NULL)
	return;

    element = (Element) attributes.user_data;
    drawn = (Drawn) element -> aux;
    if (drawn -> type != DrawnElement)
	return;

    EditElementInfo (element);
}


void EditElementAP ( )
{
    char          *status;
    struct element dummy;
    Item           found;


    if ((status = GetTextNumber (&dummy.number)) != NULL)
	return;

    found = TreeSearch (problem.element_tree, (Item) &dummy);
    if (found == NULL) {
	error ("Element %d does not exist.", dummy.number);
	return;
    }

    EditElementInfo (found);
}


void EditElementNumber ( )
{
    SetEditMode ( );
    SetShellTitle (error_dialog -> shellwidget, "Edit Element");
    ChangeStatusLine ("Select element:", True);

    XtRemoveAllCallbacks (drawing, XtNbuttonCallback);
    XtAddCallback (drawing, XtNbuttonCallback, EditElementCB, NULL);

    XtOverrideTranslations (entry,
	XtParseTranslationTable ("<Key>Return: EditElementAP()"));
}


static Node     nodes [20];
static unsigned current_node;
static unsigned num_nodes = 0;


static char *ordinals [ ] = {"", "first", "second", "third", "fourth", "fifth",
			"sixth", "seventh", "eighth", "ninth", "tenth",
			"eleventh", "twelfth", "thirteenth", "fourteenth"};


void AbortAddElement ( )
{
    if (num_nodes == 0)
	SetNormalMode ( );
    else {
	current_node = 1;
	ChangeStatusLine ("Aborted.  Select first node:", True);
    }
}


static Boolean first_time;

void DoAddElement (node)
    Node node;
{
    static char      message [80];
    unsigned         i;
    unsigned         max;
    Element          element;


    nodes [current_node] = node;

    if (current_node != num_nodes) {
	sprintf (message, "Selected node%s", current_node == 1 ? "" : "s");
	for (i = 1; i <= current_node; i ++)
	    sprintf (message + strlen (message), "%s %d", i == 1 ? "" : ",",
				nodes [i] != NULL ? nodes [i] -> number : 0);
	sprintf (message + strlen (message), ".  Select %s node:",
						ordinals [++ current_node]);
	ChangeStatusLine (message, True);
	changeflag = True;
	return;
    }


    element = (Element) TreeMaximum (problem.element_tree);
    max = element != NULL ? element -> number : 0;

    element = CreateElement (max + 1, active_definition);
    element -> material = active_material;
    for (i = 1; i <= num_nodes; i ++)
	element -> node [i] = nodes [i];

    DrawElement (element, first_time);

    current_node = 1;
    sprintf (message, "Added element %d.  Select first node:", max + 1);
    ChangeStatusLine (message, True);

    if (XtIsRealized (elementShell) == True)
	GetElementInformation (element);

    first_time = False;
}


void AddElementAP ( )
{
    char       *status;
    struct node dummy;
    Item        found;


    if ((status = GetTextNumber (&dummy.number)) != NULL)
	return;

    if (dummy.number == 0) {
	DoAddElement (NULL);
	return;
    }

    found = TreeSearch (problem.node_tree, (Item) &dummy);
    if (found == NULL) {
	error ("Node %d does not exist.", dummy.number);
	return;
    }

    DoAddElement (found);
}


void AddElementCB (w, client_data, call_data)
    Widget    w;
    XtPointer client_data;
    XtPointer call_data;
{
    Node             node;
    Drawn            drawn;
    Figure           figure;
    DrawingReport   *report;
    FigureAttributes attributes;


    report = (DrawingReport *) call_data;
    if (report -> event -> type != ButtonPress)
	return;

    if (report -> event -> xbutton.button == 3)
	QuitEdit ( );

    if (report -> event -> xbutton.button != 1)
	return;

    figure = DW_FindFigure (w, report -> unsnapped.x, report -> unsnapped.y);
    if (figure == NULL)
	return;

    DW_GetAttributes (w, figure, &attributes);
    if (attributes.user_data == NULL)
	return;

    node = (Node) attributes.user_data;
    drawn = (Drawn) node -> aux;
    if (drawn -> type != DrawnNode)
	return;

    DoAddElement (node);
}


void EditAddElement ( )
{
    SetShellTitle (error_dialog -> shellwidget, "Add Element");

    if (active_material_number == -1) {
	error ("No active material defined.");
	return;
    }

    if (active_definition_number == -1) {
	error ("No active element type defined.");
	return;
    }

    first_time = True;
    SetEditMode ( );
    ChangeStatusLine ("Select first node:", True);

    current_node = 1;
    num_nodes = active_definition -> numnodes;

    XtRemoveAllCallbacks (drawing, XtNbuttonCallback);
    XtAddCallback (drawing, XtNbuttonCallback, AddElementCB, NULL);

    AssignQuitAbort (QuitEdit, "QuitEdit", AbortAddElement, "AbortAddElement");

    XtOverrideTranslations (entry,
	XtParseTranslationTable ("<Key>Return: AddElementAP()"));
}


int DrawElement (element, first_time)
    Element element;
    Boolean first_time;
{
    Point            points [100];
    Figure           fig;
    FigureAttributes attr;
    Drawn            drawn;
    unsigned         num;
    unsigned         j;
    Item	     found;

    found = TreeInsert (problem.element_tree, element);
    if (found != (Item) element)
	return 1;

    if (first_time == True)
	if (DW_SetForeground (drawing, elt_color) == False)
	    (void) DW_SetForeground (drawing, "black");


    num = element -> definition -> shapenodes;

    if (num > 2) {
	for (j = 1; j <= num; j ++) {
	    if (element -> node [j] == NULL || element -> node [j] -> z != 0)
		break;

	    points [j - 1].x = element -> node [j] -> x;
	    points [j - 1].y = element -> node [j] -> y;
	}

	if (j > num) {
	    points [num].x = element -> node [1] -> x;
	    points [num].y = element -> node [1] -> y;
	    fig = DW_DrawPolygon (drawing, True, points, num + 1);
	} else
	    fig = NULL;

    } else {
	if (element -> node [1] != NULL && element -> node [2] != NULL)
	    if (element -> node [1] -> z == 0 && element -> node [2] -> z == 0)
		fig = DW_DrawLine (drawing, element -> node [1] -> x,
			element -> node [1] -> y, element -> node [2] -> x,
			element -> node [2] -> y);
	    else
		fig = NULL;
	else
	    fig = NULL;
    }

    drawn = (Drawn) XtNew (struct drawn);
    drawn -> type = DrawnElement;
    drawn -> figure = fig;
    drawn -> label = NULL;
    element -> aux = (char *) drawn;

    if (fig != NULL) {
	attr.user_data = (char *) element;
	DW_SetAttributes (drawing, fig, DW_FigureUserData, &attr);
    }

    for (j = 1; j <= element -> definition -> numnodes; j ++)
	if (element -> node [j] != NULL) {
	    if (element -> node [j] -> aux == NULL) {
		drawn = (Drawn) XtNew (struct drawn);
		drawn -> type = DrawnNode;
		drawn -> figure = NULL;
		drawn -> label = NULL;
		drawn -> ref_count = 0;
		element -> node [j] -> aux = (char *) drawn;
	    }

	    drawn = (Drawn) element -> node [j] -> aux;
	    drawn -> ref_count ++;
	    DW_RaiseFigure (drawing, drawn -> figure);
	    DW_RaiseFigure (drawing, drawn -> label);
	}

    return 0;
}
