/*
    This file is part of the FElt finite element analysis package.
    Copyright (C) 1993 Jason I. Gobat and Darren C. Atkinson

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

# include <stdio.h>
# include <string.h>
# include <X11/Intrinsic.h>
# include "fe.h"
# include "forms.h"
# include "dialog.h"
# include "problem.h"
# include "globals.h"
# include "Drawing.h"
# include "vfe.h"
# include "util.h"
# include "objects.h"
# include "procedures.h"
# include "error.h"

static int count;


static int SetConstraintNames (item)
    Item item;
{
    if (active_constraint == (Constraint) item)
	active_constraint_number = count;

    constraint_names [count ++] = ((Constraint) item) -> name;
    return 0;
}


void GatherConstraintNames ( )
{
    int size;


    if ((size = TreeSize (problem.constraint_tree)) <= 0)
	return;

    count = 0;
    size *= sizeof (String);
    constraint_names = (String *) XtRealloc ((char *) constraint_names, size);
    (void) TreeSetIterator (problem.constraint_tree, SetConstraintNames);
    (void) TreeIterate (problem.constraint_tree);
}


void SetActiveConstraint ()
{
    int               result, size;
    struct constraint dummy;


    SetShellTitle (constraint_list -> shellwidget, "Set Active Constraint");
    if ((size = TreeSize (problem.constraint_tree)) <= 0) {
	error ("No constraints defined.");
	return;
    }

    SetListItems (constraint_list, constraint_names, size);
    result = GetListFormValue (constraint_list, active_constraint_number);

    if (result > -1) {
	dummy.name = constraint_names [result];
	active_constraint = (Constraint) TreeSearch (problem.constraint_tree, &dummy);
	active_constraint_number = result;
	changeflag = True;
    }
}


void GetConstraintDefinition ()
{
    Boolean 		result[6];
    Constraint	        constraint;
    extern ToggleRecord constraint_entries;
    unsigned		i, status;
    String		c_name;
    Item		found;

    SetShellTitle (constraint_form -> shellwidget, "Define Constraint");
    status = GetToggleFormValues (constraint_form, constraint_entries, 
                                  result, &c_name);

    if (status) {
	if (strlen (c_name) > 0) {
	    constraint = CreateConstraint (XtNewString (c_name));

	    for (i = 1; i <= 6; i ++)
		constraint -> constraint [i] = result [i - 1] == True;

	    found = TreeInsert (problem.constraint_tree, (Item) constraint);
	    if (found != (Item) constraint) {
		error ("Constraint %s already exists.", constraint -> name);
		DestroyConstraint (constraint);
		return;
	    }

	    active_constraint = constraint;
	    GatherConstraintNames ( );
	    changeflag = True;
	} else
	    error ("No constraint name specified.");
    }
}


void EditConstraintInfo ()
{
    int			i, status, size;
    Boolean		result [6];
    extern ToggleRecord	constraint_entries;
    Constraint		constraint;
    static Boolean	suggestion [7];
    static char         name [256];
    struct constraint	dummy;
    String		c_name;
    Item		found;


    SetShellTitle (constraint_list -> shellwidget, "Edit Constraint");
    if ((size = TreeSize (problem.constraint_tree)) <= 0) {
	error ("No constraints defined.");
	return;
    }

    SetListItems (constraint_list, constraint_names, size);
    status = GetListFormValue (constraint_list, active_constraint_number);

    if (status == -1)
	return;

    dummy.name = constraint_names [status];
    constraint = (Constraint) TreeSearch (problem.constraint_tree, &dummy);

    strcpy (name, dummy.name);
    for (i = 1; i <= 6; i ++)
	suggestion [i - 1] = constraint -> constraint [i] ? True : False;

    SetToggleStates (constraint_form, constraint_entries, suggestion, name);

    SetShellTitle (constraint_form -> shellwidget, "Edit Constraint");
    status = GetToggleFormValues (constraint_form, constraint_entries, 
                                  result, &c_name);

    if (status) {
	if (strlen (c_name) > 0) {
	    if (strcmp (c_name, constraint -> name))
		constraint = CreateConstraint (XtNewString (c_name));

	    for (i = 1; i <= 6; i ++)
		constraint -> constraint [i] = result [i - 1] == True;

	    found = TreeInsert (problem.constraint_tree, (Item) constraint);
	    if (found != (Item) constraint) {
		error ("Constraint %s already exists.", constraint -> name);
		DestroyConstraint (constraint);
		return;
	    }

	    active_constraint = constraint;
	    GatherConstraintNames ( );
	    changeflag = True;
	} else
	    error("No constraint name specified.");
    }
}


static Constraint constraint;
static unsigned number;

static int ConstraintRef (item)
    Item item;
{
    if (((Node) item) -> constraint == constraint) {
	number = ((Node) item) -> number;
	return 1;
    }

    return 0;
}


void DeleteConstraint ()
{
    int			result, size;
    struct constraint	dummy;


    SetShellTitle (constraint_list -> shellwidget, "Delete Constraint");
    if ((size = TreeSize (problem.constraint_tree)) <= 0) {
	error ("No constraints defined.");
	return;
    }

    SetListItems (constraint_list, constraint_names, size);
    result = GetListFormValue (constraint_list, -1);

    if (result > -1) {
	dummy.name = constraint_names [result];
	constraint = (Constraint) TreeSearch (problem.constraint_tree, &dummy);
	(void) TreeSetIterator (problem.node_tree, ConstraintRef);
	if (TreeIterate (problem.node_tree)) {
	    error ("Constraint %s is still in use by node %d.", dummy.name, number);
	    return;
	}

	DestroyConstraint ((Constraint) TreeDelete (problem.constraint_tree, &dummy));
	if (active_constraint_number == result)
	    active_constraint_number = -1;

	GatherConstraintNames ( );
	changeflag = True;
    }
}


void DoApplyConstraint (node)
    Node node;
{
    static char message [40];


    node -> constraint = active_constraint;

    if (XtIsRealized (nodeShell))
	GetNodeInformation (node);

    sprintf (message, "Applied to node %d.  Select node:", node -> number);
    ChangeStatusLine (message, True);
    changeflag = True;
}


void ApplyConstraintCB (w, client_data, call_data)
    Widget    w;
    XtPointer client_data;
    XtPointer call_data;
{
    DrawingReport   *report;
    FigureAttributes attributes;
    Figure           figure;
    Drawn	     drawn;
    Node             node;


    report = (DrawingReport *) call_data;

    if (report -> event -> type != ButtonPress)
	return;

    if (report -> event -> xbutton.button == 3)
	QuitEdit ( );

    if (report -> event -> xbutton.button != 1)
	return;

    figure = DW_FindFigure (w, report -> unsnapped.x, report -> unsnapped.y);

    if (figure == NULL)
	return;

    DW_GetAttributes (w, figure, &attributes);
    if (attributes.user_data == NULL)
	return;

    node = (Node) attributes.user_data;
    drawn = (Drawn) node -> aux;
    if (drawn -> type != DrawnNode)
	return;

    DoApplyConstraint (node);
}


void ApplyConstraintAP ( )
{
    char       *status;
    struct node dummy;
    Item        found;


    if ((status = GetTextNumber (&dummy.number)) != NULL)
	return;

    found = TreeSearch (problem.node_tree, (Item) &dummy);
    if (found == NULL) {
	error ("Node %d does not exist.", dummy.number);
	return;
    }

    DoApplyConstraint (found);
}


void EditApplyConstraint ( )
{

    SetShellTitle (error_dialog -> shellwidget, "Apply Constraint");
    if (active_constraint_number == -1) {
	error ("No active constraint defined.");
	return;
    }

    SetEditMode ( );
    ChangeStatusLine ("Select node:", True);

    XtRemoveAllCallbacks (drawing, XtNbuttonCallback);
    XtAddCallback (drawing, XtNbuttonCallback, ApplyConstraintCB, NULL);

    XtOverrideTranslations (entry,
	XtParseTranslationTable ("<Key>Return: ApplyConstraintAP()"));
}
