/*
 * Copyright (c) 1991, 1993 
 * Regents of the University of Michigan.  All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <malloc.h>
#ifndef __STDC__
#include <memory.h>
#endif
#include <lber.h>
#include <ldap.h>
#include "ud.h"

#ifdef DEBUG
extern int debug;
#endif

struct entry2 Entry;
extern LDAP *ld;

/*
 *  When displaying entries, display only these attributes, and in this
 *  order.
 */
static char *person_attr_print_order[] = {
	"cn",
	"mail",
	"facsimileTelephoneNumber",
	"telephoneNumber",
	"postalAddress",
	"homePhone",
	"homePostalAddress",
	"title",
	"uid",
	"multiLineDescription",
	"memberOfGroup",
	NULL
};

static char *group_attr_print_order[] = {
	"cn",
	"facsimileTelephoneNumber",
	"telephoneNumber",
	"postalAddress",
	"multiLineDescription",
	"joinable",
	"associatedDomain",
	"owner",
	"rfc822ErrorsTo",
	"rfc822RequestsTo",
	"member",
	"mail",
	NULL
};

static char *known_attributes[] = {
	"memberOfGroup",
	"acl",
	"cn",
	"title",
	"postalAddress",
	"telephoneNumber",
	"mail",
	"homePhone",
	"homePostalAddress",
	"objectClass",
	"multiLineDescription",
	"krbName",
	"description",
	"facsimileTelephoneNumber",
	"uid",
	"userPassword",
	"noBatchUpdates",
	"joinable",
	"associatedDomain",
	"owner",
	"member",
	"rfc822ErrorsTo",
	"rfc822RequestsTo",
	/*
	 *  If you add any entries to this list, be sure you increment the
	 *  value of LAST_ATTRIBUTE in ud.h.  It should be equal to the number
	 *  of lines above plus one.
	 *
	 *  Also add the output string that is to go along with this item.
	 */
	NULL };

parse_answer(s)
LDAPMessage *s;
{
	int idx;
	char **rdns;
	BerElement *cookie;
	register LDAPMessage *ep;
	register char *ap;
	void clear_entry();

#ifdef DEBUG
	if (debug & D_TRACE)
		printf("->parse_answer(%x)\n", s);
#endif

	clear_entry();
	for (ep = ldap_first_entry(ld, s); ep != NULL; ep = ldap_next_entry(ld, ep)) {
		Entry.DN = ldap_get_dn(ld, ep);
		rdns = ldap_explode_dn(Entry.DN, TRUE);
		Entry.name = *rdns;
		for (ap = ldap_first_attribute(ld, ep, &cookie); ap != NULL; ap = ldap_next_attribute(ld, ep, cookie)) {

#ifdef DEBUG
			if (debug & D_PARSE)
				printf("parsing ap = %s\n", ap);
#endif
			if ((idx = attr_to_index(ap)) < 0) {
				printf("  Unknown attribute \"%s\"\n", ap);
				continue;
			}
			add_value(&(Entry.attrs[idx]), ep, ap);
		}
	}
}

add_value(attr, ep, ap)
struct attribute *attr;
LDAPMessage *ep;
char *ap;
{
	register int i = 0;
	char **vp, **tp, **avp;

#ifdef DEBUG
	if (debug & D_TRACE)
		printf("->add_value(%x, %x, %s)\n", attr, ep, ap);
#endif
	vp = (char **) ldap_get_values(ld, ep, ap);

	/*
	 *  Fill in the attribute structure for this attribute.  This
	 *  stores away the values (using strdup()) and the count.  Terminate
	 *  the list with a NULL pointer.
	 */
	attr->number_of_values = ldap_count_values(vp);
	attr->quipu_name = strdup(ap);
	if ((attr->values = (char **) malloc((unsigned) ((attr->number_of_values + 1) * sizeof(char *)))) == (char **) NULL) {
		fatal("malloc");
		/*NOTREACHED*/
	}
	avp = attr->values;

	for (i = 1, tp = vp; *tp != NULL; i++, tp++) {
#ifdef DEBUG
		if (debug & D_PARSE)
			printf("  value #%d  %s\n", i, *tp);
#endif
		/*
		 *  The 'name' field of the Entry structure already has
		 *  has the first part of the DN copied into it.  Thus,
		 *  we don't need to save it away here again.  Also, by
		 *  tossing it away here, we make printing this info out
		 *  a bit easier later.
		 */
		if (!strcmp(ap, "cn") && !strcmp(*tp, Entry.name)) {
			attr->number_of_values--;
			continue;
		}
		*avp++ = strdup(*tp);
	}
	*avp = NULL;
	ldap_value_free(vp);
}

print_an_entry(type)
char *type;
{
	int n = 0, i, idx;
	char is_a_group, **order;
	char *sub_list[MAX_VALUES * 25], buf[RESP_SIZE];

#ifdef DEBUG
	if (debug & D_TRACE)
		printf("->print_an_entry(%s)\n", type);
#endif
	printf(" \"%s\"\n", Entry.name);
	
	/*
	 *  If the entry is a group, find all of the subscribers to that
	 *  group.  A subscriber is an entry that *points* to a group entry,
	 *  and a member is an entry that is included as part of a group
	 *  entry.
	 *
	 *  We also need to select the appropriate output format here.
	 */
	is_a_group = isgroup();
	if (is_a_group) {
		order = (char **) group_attr_print_order;
		n = find_all_subscribers(sub_list, Entry.DN);
#ifdef DEBUG
		if (debug & D_PRINT)
			printf(" Group \"%s\" has %d subscribers\n", 
								Entry.name, n);
#endif
	}
	else
		order = (char **) person_attr_print_order;

	for (i = 0; order[i] != NULL; i++) {
		idx = attr_to_index(order[i]);
#ifdef DEBUG
		if (debug & D_PRINT) {
			printf("  ATTR #%2d = %s [%s] (%d values)\n", i + 1,
				Entry.attrs[idx].output_string,
				Entry.attrs[idx].quipu_name,
				Entry.attrs[idx].number_of_values);
		}
#endif
		if (idx < 0)
			continue;
		if (Entry.attrs[idx].number_of_values == 0)
			continue;
		if (!strcmp(order[i], "memberOfGroup"))
			print_DN(Entry.attrs[idx]);
		else if (!strcmp(order[i], "member"))
			print_DN(Entry.attrs[idx]);
		else if (!strcmp(order[i], "owner"))
			print_DN(Entry.attrs[idx]);
		else
			print_values(Entry.attrs[idx]);
	}

	/*
	 *  If it is a group, then we should print the subscriber list (if
	 *  there are any).  If there are a lot of them, prompt the user
	 *  before printing them.
	 */
	if (is_a_group && (n > 0)) {
		char *label = "Subscribers:         ";

		if (n > TOO_MANY_TO_PRINT) {
			printf("  There are %d subscribers.  Print them? ", n);
			fflush(stdout);
			fetch_buffer(buf, sizeof(buf), stdin);
			if (!((buf[0] == 'y') || (buf[0] == 'Y')))
				return;
		}
		format2(ldap_dn2ufn(sub_list[n - 1]), label, (char *) NULL, 2, 
						2 + strlen(label) + 1, 80); 
		for (n--; n > 0; n--)
			format2(ldap_dn2ufn(sub_list[n - 1]), (char *) NULL, 
				(char *) NULL, 2 + strlen(label), 
				2 + strlen(label) + 2, 80); 
	}

	return;
}

#define OUT_LABEL_LEN	20

/* prints the values associated with an attribute */
print_values(A)
struct attribute A;
{
	register int i, k;
	register char *cp, **vp;
	char out_buf[BUFSIZ], *padding = NULL;
	int lead;

#ifdef DEBUG
	if (debug & D_TRACE)
		printf("->print_values(%x)\n", A);
#endif
	if (A.number_of_values == 0)
		return;
	if ((vp = A.values) == NULL)
		return;

	/*
	 *  Pad out the output string label so that it fills the
	 *  whole field of length OUT_LABEL_LEN.
	 */
	out_buf[0] = '\0';
	i = OUT_LABEL_LEN - strlen(A.output_string);
	if (i < 0) {
		printf("Output string for \"%s\" is too long.  Maximum length is %d characters\n", A.quipu_name, OUT_LABEL_LEN);
		return;
	}
	if (isgroup() && !strcmp(A.quipu_name, "mail") && (Entry.attrs[attr_to_index("member")].number_of_values == 0)) {
		A.output_string = "Members";
		i = OUT_LABEL_LEN - strlen(A.output_string);
		if ((padding = (char *) malloc(i + 1)) == NULL) {
			fatal("malloc");
			/*NOTREACHED*/
		}
		(void) memset(padding, ' ', i);
		*(padding + i) = '\0';
		sprintf(out_buf, "%s:%s", A.output_string, padding);
	}
	else if (!(isgroup() && !strcmp(A.quipu_name, "mail") && (Entry.attrs[attr_to_index("member")].number_of_values > 0))) {
		if ((padding = (char *) malloc(i + 1)) == NULL) {
			fatal("malloc");
			/*NOTREACHED*/
		}
		(void) memset(padding, ' ', i);
		*(padding + i) = '\0';
		sprintf(out_buf, "%s:%s", A.output_string, padding);
	}
	/*
	 *  If this happens to be a group, then do not print the output
	 *  string if we have already printed out some members.
	 */
	else if (isgroup() && !strcmp(A.quipu_name, "mail") && (Entry.attrs[attr_to_index("member")].number_of_values > 0)) {
		if ((padding = (char *) malloc(OUT_LABEL_LEN + 2)) == NULL) {
			fatal("malloc");
			/*NOTREACHED*/
		}
		(void) memset(padding, ' ', OUT_LABEL_LEN + 1);
		*(padding + OUT_LABEL_LEN + 1) = '\0';
		sprintf(out_buf, "%s", padding);
	}
	lead = strlen(out_buf) + 2;

	printf("  %s", out_buf);
	for (i = 0; *vp != NULL; i++, vp++) {
		if (i > 0) {
			if (!strncmp(A.output_string, "Home a", 6) || !strncmp(A.output_string, "Business a", 10)) {
				printf("  %s", out_buf);
			}
			else {
				for (k = lead; k > 0; k--)
					putchar(' ');
			}
		}
		for (cp = *vp; *cp != '\0'; cp++) {
			switch (*cp) {
			case '$' :
				if (!strncmp(A.output_string, "Home a", 6) || !strncmp(A.output_string, "Business a", 10) || !strcmp(A.quipu_name, "multiLineDescription")) {
					putchar('\n');
					for (k = lead; k > 0; k--)
						putchar(' ');
					while (isspace(*(cp + 1)))
						cp++;
				}
				else
					putchar(*cp);
				break;
			case '\n' :
				putchar('%');
				putchar('\n');
				break;
			default:
				putchar(*cp);
			}
		}
		putchar('\n');
	}
	if (padding != NULL)
		(void) free(padding);
	return;
}

/* prints the DN's associated with an attribute */
print_DN(A)
struct attribute A;
{
	int i, lead;
	register char **vp;
	char out_buf[BUFSIZ], *padding = NULL;

#ifdef DEBUG
	if (debug & D_TRACE)
		printf("->print_DN(%x)\n", A);
#endif
	if (A.number_of_values == 0)
		return;
	/*
	 *  Pad out the output string label so that it fills the
	 *  whole field of length OUT_LABEL_LEN.
	 */
	i = OUT_LABEL_LEN - strlen(A.output_string);
	if (i > 0) {
		if ((padding = (char *) malloc(i + 1)) == NULL) {
			fatal("malloc");
			/*NOTREACHED*/
		}
		(void) memset(padding, ' ', i);
		*(padding + i) = '\0';
		sprintf(out_buf, "%s:%s", A.output_string, padding);
	}
	lead = strlen(out_buf) + 2;

	/*  print out the values (which are Distinguished Names) */
	vp = A.values;
	format2(ldap_dn2ufn(*vp), out_buf, (char *) NULL, 2, lead + 1, 80); 
	for (vp++; *vp != NULL; vp++) {
		format2(ldap_dn2ufn(*vp), (char *) NULL, (char *) NULL, lead, 
			lead + 1, 80); 
	}
	return;
}

void clear_entry()
{
	register int i;
	register char **cpp;

#ifdef DEBUG
	if (debug & D_TRACE)
		printf("->clear_entry()\n");
	if ((debug & D_PRINT) && (Entry.name != NULL))
		printf(" Clearing entry \"%s\"\n", Entry.name);
#endif
	if (Entry.DN != NULL)
		(void) free(Entry.DN);
	if (Entry.name != NULL)
		(void) free(Entry.name);
	Entry.may_join = FALSE;
	Entry.subscriber_count = -1;
	Entry.DN = Entry.name = NULL;

	/*  clear all of the values associated with all attributes */
	for (i = 0; i < LAST_ATTRIBUTE; i++) {
		cpp = Entry.attrs[i].values;
#ifdef DEBUG
		if (debug & D_PRINT)
			printf(" Clearing attribute \"%s\"\n", 
				Entry.attrs[i].quipu_name);
#endif
		if (cpp == NULL)
			continue;
		for ( ; *cpp != NULL; cpp++) {
#ifdef DEBUG
			if (debug & D_PRINT)
				printf(" value = %s\n", *cpp);
#endif
			(void) free(*cpp);
		}
		Entry.attrs[i].number_of_values = 0;
		Entry.attrs[i].values = (char **) NULL;
		/*
		 *  Note:  We do not clear either of the char * fields
		 *  since they will always be applicable.
		 */
	}
}

attr_to_index(s)
char *s;
{
	register int i;

	for (i = 0; known_attributes[i] != NULL; i++)
		if (!strcasecmp(known_attributes[i], s))
			return(i);
	return(-1);
}

void initialize_attribute_strings()
{
	register int i;
	extern struct entry2 Entry;

	Entry.attrs[ 0].output_string = "Subscribed to";
	Entry.attrs[ 1].output_string = "Access Control";
	Entry.attrs[ 2].output_string = "Aliases";
	Entry.attrs[ 3].output_string = "Title";
	Entry.attrs[ 4].output_string = "Business address";
	Entry.attrs[ 5].output_string = "Business phone";
	Entry.attrs[ 6].output_string = "E-mail address";
	Entry.attrs[ 7].output_string = "Home phone";
	Entry.attrs[ 8].output_string = "Home address";
	Entry.attrs[ 9].output_string = "Object class";
	Entry.attrs[10].output_string = "Description";
	Entry.attrs[11].output_string = "Kerberos name";
	Entry.attrs[12].output_string = "Brief description";
	Entry.attrs[13].output_string = "Fax number";
	Entry.attrs[14].output_string = "Uniqname";
	Entry.attrs[15].output_string = "Password";
	Entry.attrs[16].output_string = "No batch updates";
	Entry.attrs[17].output_string = "Joinable flag";
	Entry.attrs[18].output_string = "Associated domain";
	Entry.attrs[19].output_string = "Owner";
	Entry.attrs[20].output_string = "Members";
	Entry.attrs[21].output_string = "Errors to";
	Entry.attrs[22].output_string = "Requests to";
	Entry.attrs[23].output_string = "* LAST ATTRIBUTE *";

	for (i = 0; known_attributes[i] != NULL; i++)
		Entry.attrs[i].quipu_name = known_attributes[i];
}
