/*
 * Copyright (c) 1991, 1992, 1993 
 * Regents of the University of Michigan.  All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 *
 * The University of Michigan would like to thank the following people for
 * their contributions to this piece of software:
 *
 *	Robert Urquhart    <robert@sfu.ca>
 *	Simon Fraser University, Academic Computing Services
 */

#include <stdio.h>
#include <sys/types.h>
#include <pwd.h>
#include <string.h>
#ifndef DOS
#include <sgtty.h>
#endif
#include <ctype.h>
#include <signal.h>
#include <setjmp.h>
#if defined(NeXT)
#include <stdlib.h>
#endif
#include <memory.h>
#include <lber.h>
#include <ldap.h>
#include "ud.h"

#ifndef lint
char copyright[] =
"@(#) Copyright (c) 1991, 1992, 1993 Regents of the University of Michigan.\nAll rights reserved.\n";
#endif

/*
 *  Used with change_base() to indicate which base we are changing.
 */
#define BASE_SEARCH     0
#define BASE_GROUPS     1

#define	iscom(x)	(!strncasecmp(x, cmd, strlen(cmd)))

static char server[BUFSIZ]; 	/* LDAP server name */

char *default_search_algorithm[] = { 
	"[1] uid=$0", 		/* try it as a uid if 1 field */
	"[1] sn=$0",		/* try it as a surname if 1 field */
	"cn=$0",		/* try it as a common name */
	"[1] sn~=$0",		/* try it as an inexact surname if 1 field */
	"(&(cn~=$0)(sn=$NF))",	/* try it as an inexact common name */
	"cn~=$0",		/* try it as an inexact common name */
	NULL 
};

static int ldap_port = LDAP_PORT;


char *default_bind_object = UD_USER;

static char *config_file = CONFIG_FILE;

char *search_algorithm[MAX_SEARCH_PATH];

char *group_base;		/* place in X.500 tree where groups are */
char *search_base;		/* place in X.500 tree where searches start */

extern char me[];		/* string containing result of I AM */
static jmp_buf env;		/* spot to jump to on an interrupt */
int lpp;			/* lines per page */
#ifdef DEBUG
int debug;			/* debug flag */
#endif
int verbose;			/* 1 if verbose mode on */
LDAP *ld;			/* LDAP descriptor */

main(argc, argv)
int argc;
char *argv[];
{
	extern char Version[];			/* version number */
	extern char *optarg;			/* for parsing argv */
	register int c;				/* for parsing argv */
	register char *cp;			/* for parsing Version */
	extern void initialize_attribute_strings();

#ifdef notdef
	if ( !isatty( 1 )) {
	    chdir( "/usr/tmp" );
	}
#endif /* notdef */

	/*  handle argument list */
	while ((c = getopt(argc, argv, "d:Df:l:p:s:u:v")) != -1) {
		switch (c) {
		case 'l' :
#ifdef LDAP_DEBUG
			ldap_debug = (int) strtol(optarg, (char **) NULL, 0);
#endif
		case 'd' :
#ifdef DEBUG
			debug = (int) strtol(optarg, (char **) NULL, 0);
#endif
			break;
		case 's' :
			strcpy(server, optarg);
			break;
		case 'f' :
			config_file = optarg;
			break;
		case 'p' :
			ldap_port = atoi(optarg);
			break;
		case 'u' :
			default_bind_object = strdup(optarg);
			break;
		case 'v' :
			verbose = 1;
			break;
		case 'D' :
			printf("\n\n  Debug flag values\n\n");
			printf("    1  function trace\n");
			printf("    2  find() information\n");
			printf("    4  group information\n");
			printf("    8  mod() information\n");
			printf("   16  parsing information\n");
			printf("   32  output information\n");
			printf("   64  authentication information\n\n");
			format("These are masks, and may be added to form multiple debug levels.  For example, '-d 35' would perform a function trace, print out information about the find() function, and would print out information about the output routines too.", 75, 2);
			exit(0);
		default:
			fprintf(stderr, "Usage: %s [-d debug-level] [-l ldap-debug-level] [-s server] [-p port] [-v]\n", argv[0]);
			exit(-1);
			/* NOTREACHED */
		}
	}

	/* just print the first line of Version[] */
	cp = strchr(Version, '\t');
	if (cp != NULL)
		*cp = '\0';
	printf(Version);
	fflush( stdout );

	initialize_client();
	initialize_attribute_strings();

	/* now tackle the user's commands */
	do_commands();
	/* NOTREACHED */
}

do_commands()
{
	LDAPMessage *mp;			/* returned by find() */
	register char *cp;			/* misc char pointer */
	register char *ap;			/* misc char pointer */
	static char buf[BUFSIZ / 8];		/* for prompting */
	static char cmd[BUFSIZ];		/* holds the command */
	static char input[BUFSIZ];		/* buffer for fgets() */
	extern char *alias();
	extern LDAPMessage *find();
	extern void purge_group(), add_group(), remove_group(), x_group();

#ifdef DEBUG
	if (debug & D_TRACE)
		printf("->do_commands()\n");
#endif
	if (verbose) 
		printf("\n  Enter a command.  If you need help, type 'h' or '?' and hit RETURN.\n\n");
	/* jump here on an interrupt */
	(void) setjmp(env);
	for (;;) {
		printf("* ");
		fflush(stdout);
		cp = input;
		fetch_buffer(input, sizeof(input), stdin);
		if (*input == '\0') {
			putchar('\n');
			continue;
		}
		while (isspace(*cp))
			cp++;	
		ap = cmd;
		if (memset(cmd, '\0', sizeof(cmd)) == NULL)
			fatal("memset");
		while (!isspace(*cp) && (*cp != '\0'))
			*ap++ = *cp++;
		if (iscom("status"))
			status();
		else if (iscom("stop") || iscom("quit"))
			break;
		else if (iscom("cb") || iscom("cd") || iscom("moveto")) {
			while (isspace(*cp) && (*cp != '\0')) 
				cp++;
			if (!strncasecmp(cp, "base", 4))
				cp += 4;
			while (isspace(*cp) && (*cp != '\0')) 
				cp++;
			if (*cp == '\0')
				cp = NULL;
			change_base(BASE_SEARCH, &search_base, cp);
		}
		else if (iscom("groupbase")) {
			while (isspace(*cp))
				cp++;
			if (*cp == '\0')
				cp = NULL;
			change_base(BASE_GROUPS, &group_base, cp);
		}
		else if (iscom("find") || iscom("display") || iscom("show")) {
			while (isspace(*cp))
				cp++;
			while (*cp == '\0') {
				printf("  Find who? ");
				fflush(stdout);
				fetch_buffer(buf, sizeof(buf), stdin);
				if ((buf != NULL) && (buf[0] != '\0'))
					cp = buf;
			}
			if ((cp = alias(cp)) == NULL)
				continue;
			if ((mp = find(cp)) != NULL) {
				parse_answer(mp);
				print_an_entry((char *) NULL);
				ldap_msgfree(mp);
			}
			else
				printf(" Could not find \"%s\".\n", cp);
		}
		else if (iscom("dump")) {
			while (isspace(*cp))
				cp++;
			while (*cp == '\0') {
				printf("  Find who? ");
				fflush(stdout);
				fetch_buffer(buf, sizeof(buf), stdin);
				if ((buf != NULL) && (buf[0] != '\0'))
					cp = buf;
			}
			if ((cp = alias(cp)) == NULL)
				continue;
			if ((mp = find(cp)) != NULL) {
				parse_answer(mp);
				print_an_entry("groups too");
				ldap_msgfree(mp);
			}
			else
				printf(" Could not find \"%s\".\n", cp);
		}
		else if (iscom("modify") || iscom("change") || iscom("alter")) {
			while (isspace(*cp))
				cp++;
			(void) modify(alias(cp));
		}
		else if (iscom("bind")) {
			while (isspace(*cp))
				cp++;
			(void) auth(alias(cp), (char *) NULL);
		}
		else if ((cmd[0] == '?') || iscom("help")) {
			while (isspace(*cp))
				cp++;
			print_help(cp);
		}
		else if (iscom("iam")) {
			while (isspace(*cp))
				cp++;
			iam(cp);
		}
		else if (iscom("join") || iscom("subscribe")) {
			while (isspace(*cp))
				cp++;
			(void) x_group(G_JOIN, cp);
		}
		else if (iscom("resign") || iscom("unsubscribe")) {
			while (isspace(*cp))
				cp++;
			(void) x_group(G_RESIGN, cp);
		}
		else if (!strncasecmp("create", cmd, strlen(cmd))) {
			while (isspace(*cp))
				cp++;
			add_group(cp);
		}
		else if (!strncasecmp("remove", cmd, strlen(cmd))) {
			while (isspace(*cp))
				cp++;
			remove_group(cp);
		}
		else if (!strncasecmp("purge", cmd, strlen(cmd))) {
			while (isspace(*cp))
				cp++;
			purge_group(cp);
		}
		else if (!strncasecmp("whoami", cmd, strlen(cmd))) {
			print_my_name();
		}
		else if (!strncasecmp("verbose", cmd, strlen(cmd))) {
			verbose = 1 - verbose;
			/*  ;-)  */
			if (verbose)
				printf("  Verbose mode has been turned on.\n");
		}
		else if (!strncasecmp("tidy", cmd, strlen(cmd))) {
			tidy_up();
		}
		else if (cmd[0] == '\0') {
			putchar('\n');
		}
		else {
			printf("  Invalid command.  Type \"help commands.\"\n");
		}
	}
	printf(" Thank you!\n");
	
	ldap_unbind(ld);
#ifdef KERBEROS
	destroy_tickets();
#endif
	exit(0);
	/* NOTREACHED */
}

status()
{
	void printbase();

#ifdef DEBUG
	if (debug & D_TRACE)
		printf("->status()\n");
#endif
	printf("  Current server is %s\n", server);
	printbase("  Search base is ", search_base);
	printbase("  Group  base is ", group_base);
	bind_info();
}

change_base(type, base, s)
int type;
char **base, *s;
{
	register char *cp, *np;			/* utility pointers */
	char **rdns;				/* for parsing */
	char *output_string;			/* for nice output */
	int num_picked;				/* # of selected base */
	int j;					/* used with num_picked */
	int i = 1;				/* index into choices array */
	int matches;				/* # of matches found */
	int rest = 1;				/* # left to display */
	char tmp[BUFSIZ];			/* temporary buffer */
	static char *choices[BUFSIZ];		/* bases from which to choose */
	static char resp[16];			/* for prompting user */
	static char buf[BUFSIZ / 8];
	void printbase();
	static char *attrs[] = { "objectClass", NULL };
	LDAPMessage *mp;			/* results from a search */
	LDAPMessage *ep;			/* for going thru bases */
	extern char * friendly_name();

#ifdef DEBUG
	if (debug & D_TRACE)
		printf("->change_base(%s, %s)\n", s, s);
#endif
	/*
	 *  If s is NULL we need to prompt the user for an argument.
	 */
	while (s == NULL) {
		if (verbose) {
			printf("  You need to specify how the base is to be changed.  Valid choices are:\n");
			printf("     ?       - list the choices immediately below this level\n");
			printf("     ..      - move up one level in the Directory tree\n");
			printf("     root    - move to the root of the Directory tree\n");
			printf("     default - move to the default level built into this program\n");
			printf("     <entry> - move to the entry specified\n");
		}
		printf("  Change base to? ");
		fflush(stdout);
		fetch_buffer(buf, sizeof(buf), stdin);
		if ((buf != NULL) && (buf[0] != '\0'))
			s = buf;
	}

	/* set the output string */
	if (type == BASE_SEARCH)
		output_string = "  Search base is now ";
	else if (type == BASE_GROUPS)
		output_string = "  Group base is now ";

	if (!strcasecmp(s, "root")) {
		*base = NULL;
		printbase("  Search base is ", *base);
		return;
	}

	/*
	 *  User wants to ascend one level in the X.500 tree.
	 *  Easy:  Just strip off the first element of the
	 *  current search base, unless it's the root, in
	 *  which case we just do nothing.
	 */
	if (!strcasecmp(s, "..")) {
		if (*base != NULL) {
			cp = np = *base;
			cp = strchr(cp, '=');
			cp++;
			if ((cp = strchr(cp, '=')) == NULL)	/* one elem */
				*base = NULL;
			else {
				while(!isspace(*cp))
					cp--;
				cp++;
				while (*cp != '\0')
					*np++ = *cp++;
				*np = '\0';
			}
			printbase(output_string, *base);
		}
		else
			printf("  You are already at the root\n");
		return;
	}

	/* user wants to see what is directly below this level */
	if (*s == '?') {
		/*
		 *  Fetch the list of entries directly below this level.
		 *  Once we have the list, we will print it for the user, one
		 *  screenful at a time.  At the end of each screen, we ask
		 *  the user if they want to see more.  They can also just
		 *  type a number at that point too.
		 */
		if (ldap_search_s(ld, *base, LDAP_SCOPE_ONELEVEL, "(|(objectClass=quipuNonLeafObject)(objectClass=externalNonLeafObject))", attrs, FALSE, &mp) != LDAP_SUCCESS) {
			if ((ld->ld_errno == LDAP_TIMELIMIT_EXCEEDED) ||
			    (ld->ld_errno == LDAP_SIZELIMIT_EXCEEDED)) {
				if (verbose) {
					printf("  Your query was too general and a limit was exceeded.  The results listed\n");
					printf("  are not complete.  You may want to try again with a more refined query.\n\n");
				}
				else
					printf("  Time or size limit exceeded.  Partial results follow.\n\n");
			} else {
				ldap_perror(ld, "ldap_search_s");
				return;
			}
		}
		if ((matches = ldap_count_entries(ld, mp)) < 1) {
			printf("  There is nothing below this level.\n");
			(void) ldap_msgfree(mp);
			return;
		}
		num_picked = 0;
		printf(" There are %d choices:\n", matches);
		for (ep = ldap_first_entry(ld, mp); ep != NULL; ep = ldap_next_entry(ld, ep)) {
			/*
			 *  Put the last component of the DN into 'lastDN'.
			 *  If we are at the root level, convert any country
			 *  codes to recognizable names for printing.
			 */
			choices[i] = ldap_get_dn(ld, ep);
			rdns = ldap_explode_dn(choices[i], TRUE);
			printf(" %2d. %s\n", i, friendly_name(*rdns));
			(void) ldap_value_free(rdns);
			i++;
			if ((rest++ > (lpp - 3)) && (i < matches)) {
				printf("More? ");
				fflush(stdout);
				fetch_buffer(resp, sizeof(resp), stdin);
				if ((resp[0] == 'n') || (resp[0] == 'N'))
					break;
				else if (((num_picked = atoi(resp)) != 0) && (num_picked < i))
					break;
				rest = 1;
			}
		}
		for (;;) {
			if (num_picked != 0) {
				j = num_picked;
				num_picked = 0;
			}
			else {
				printf(" Which number? ");
				fflush(stdout);
				fetch_buffer(resp, sizeof(resp), stdin);
				j = atoi(resp);
			}
			if (j == 0) {
				(void) ldap_msgfree(mp);
				for (i = 0; i < matches; i++)
					(void) free(choices[i]);
				return;
			}
			if ((j < 1) || (j >= i))
				printf(" Invalid number\n");
			else {
				*base = strdup(choices[j]);
				printbase(output_string, *base);
				(void) ldap_msgfree(mp);
				for (i = 0; i < matches; i++)
					(void) free(choices[i]);
				return;
			}
		}
	}
	/* set the search base back to the original default value */
	else if (!strcasecmp(s, "default")) {
		if (type == BASE_SEARCH)
			*base = strdup(WHERE_SEARCHES_START);
		else if (type == BASE_GROUPS)
			*base = strdup(WHERE_GROUPS_LIVE);
		printbase(output_string, *base);
	}
	/* they typed in something -- see if it is legit */
	else {
		/* user cannot do something like 'cb 33' */
		if (atoi(s) != 0) {
			printf("  \"%s\" is not a valid search base\n", s);
			printf("  Base unchanged.\n");
			printf("  Try using 'cb ?'\n");
			return;
		}
		/* was it a fully-specified DN? */
		if (vrfy(s)) {
			*base = strdup(s);
			printbase(output_string, *base);
			return;
		}
		/* was it a RDN relative to the current base? */
		sprintf(tmp, "ou=%s, %s", s, *base);
		if (vrfy(tmp)) {
			*base = strdup(tmp);
			printbase(output_string, *base);
			return;
		}
		printf("  \"%s\" is not a valid base\n  Base unchanged.\n", s);
	}
}

initialize_client()
{
	register int i;
	int algorithm = 0;			/* # of algorithm */
	FILE *fp;				/* for config file */
	static char buffer[BUFSIZ];		/* for fgets() */
	struct passwd *pw;			/* for getting the home dir */
	register char *cp;			/* for fiddling with fgets() */
	char *term;				/* for tty set-up */
	char *config;				/* config file to use */
	static char bp[1024];			/* for tty set-up */
	extern char bound_as[];			/* who we are bound as */
	extern SIG_FN attn();			/* ^C signal handler */
	extern char *getenv();

#ifdef DEBUG
	if (debug & D_TRACE)
		printf("->initialize_client()\n");
#endif
	/*
	 *  A per-user config file has precedence over any system-wide
	 *  config file.
	 */
	if ((pw = getpwuid(geteuid())) == (struct passwd *) NULL)
		config = config_file;
	else {
		if (pw->pw_dir == NULL)
			config = config_file;
		else {
			sprintf(buffer, "%s/%s", pw->pw_dir, USER_CONFIG_FILE);
			config = buffer;
		}
	}

	/*
	 *  If there is a config file, read it.
	 *
	 *  Could have lines that look like this:
	 *
	 *	server <ip-address or domain-name>
	 *	base   <default search base>
	 *	groupbase <default place where groups are created>
	 *
	 *  There can be more than one line like this:
	 *
	 *	search <search algorithm>
	 *
	 *  There can be multiple search lines in the config file.  If
	 *  there are, then the order in which they appear is the order
	 *  in which they are used by the find() function.
	 */
	search_algorithm[0] = NULL;
	if ((fp = fopen(config, "r")) != NULL) {
		while (fgets(buffer, sizeof(buffer), fp) != NULL) {
			if (!strncasecmp(buffer, "server", 6)) {
				if (server[0] == '\0') {
					cp = buffer + 6;
					while (isspace(*cp))
						cp++;
					if ((*cp == '\0') || (*cp == '\n'))
						continue;
					(void) strncpy(server, cp, sizeof(server) - 1);
					if ((cp = strchr(server, '\n')) != NULL)
						*cp = '\0';
				}
			}
			else if (!strncasecmp(buffer, "base", 4)) {
				cp = buffer + 4;
				while (isspace(*cp))
					cp++;
				if ((*cp == '\0') || (*cp == '\n'))
					continue;
				search_base = strdup(cp);
				if ((cp = strchr(search_base, '\n')) != NULL)
					*cp = '\0';
			}
			else if (!strncasecmp(buffer, "groupbase", 9)) {
				cp = buffer + 9;
				while (isspace(*cp))
					cp++;
				if ((*cp == '\0') || (*cp == '\n'))
					continue;
				group_base = strdup(cp);
				if ((cp = strchr(group_base, '\n')) != NULL)
					*cp = '\0';
			}
			else if (!strncasecmp(buffer, "search", 6)) {
				cp = buffer + 6;
				while (isspace(*cp))
					cp++;
				if ((*cp == '\0') || (*cp == '\n'))
					continue;
				(void) strcpy(search_algorithm[algorithm], cp);
				if ((cp = strchr(search_algorithm[algorithm], '\n')) != NULL)
					*cp = '\0';
				algorithm++;
			}
			else
				fprintf(stderr, "?? -> %s\n", buffer);
		}
	}
	if (group_base == NULL)
		group_base = strdup(WHERE_GROUPS_LIVE);
	if (search_base == NULL)
		search_base = strdup(WHERE_SEARCHES_START);
	if (server[0] == '\0')
		strcpy(server, DEFAULT_SERVER);
	if (search_algorithm[0] == NULL) {
		for (i = 0; default_search_algorithm[i] != NULL; i++)
			search_algorithm[i] = default_search_algorithm[i];
		search_algorithm[i] = NULL;
	}

	/*
	 *  Set up our LDAP connection.  The values of retry and timeout
	 *  are meaningless since we will immediately be doing a null bind
	 *  because we want to be sure to use TCP, not UDP.
	 */
	if ((ld = ldap_open(server, ldap_port)) == NULL) {
		fprintf(stderr, "Sorry, the X.500 service is temporarily unavailable.\n");
		fprintf(stderr, "Please try again later.  Additional info:\n");
		perror("ldap_open");
		exit(0);
		/* NOTREACHED */
	}
	if (ldap_bind_s(ld, (char *) default_bind_object, (char *) UD_PASSWD,
	    LDAP_AUTH_SIMPLE) != LDAP_SUCCESS) {
		fprintf(stderr, "Sorry, the X.500 service is temporarily unavailable.\n");
		fprintf(stderr, "Please try again later.  Additional info:\n");
		ldap_perror(ld, "ldap_bind_s");
		exit(0);
		/* NOTREACHED */
	}
	(void) strcpy(bound_as, DEFAULT_BOUND_AS);

	/* enabled local caching of ldap results, 15 minute lifetime */
#ifdef DOS
	ldap_enable_cache( ld, 60 * 15, 100 * 1024 );	/* 100k max memory */
#else /* DOS */
	ldap_enable_cache( ld, 60 * 15, 0 );		/* no memory limit */
#endif /* DOS */

	/* terminal initialization stuff goes here */
	lpp = DEFAULT_LINES;

	(void) signal(SIGINT, attn);

#if !defined(DOS) && !defined(NOTERMCAP)
	{
	struct winsize win;			/* for tty set-up */
	extern SIG_FN chwinsz();		/* WINSZ signal handler */

	if (((term = getenv("TERM")) == NULL) || (tgetent(bp, term) <= 0))
		return;
	else {
		if (ioctl(fileno(stdout), TIOCGWINSZ, &win) < 0) {
			lpp = tgetnum("li");
		}
		else {
			if ((lpp = win.ws_row) == 0)
				lpp = tgetnum("li");
			if ((lpp <= 0) || tgetflag("hc"))
				lpp = DEFAULT_LINES;
		}
	}
	(void) signal(SIGWINCH, chwinsz);

	}
#endif
}

SIG_FN attn()
{
	fflush(stderr);
	fflush(stdout);
	printf("\n\n  INTERRUPTED!\n");
#if defined(DOS) || defined(SYSV)
	(void) signal(SIGINT, attn);
#endif
	longjmp(env, 1);
}

#if !defined(DOS) && !defined(NOTERMCAP)
SIG_FN chwinsz() 
{
	struct winsize win;

	(void) signal(SIGWINCH, SIG_IGN);
	if (ioctl(fileno(stdout), TIOCGWINSZ, &win) != -1) {
		if (win.ws_row != 0)
			lpp = win.ws_row;
	}
	(void) signal(SIGWINCH, chwinsz);
}
#endif

char * alias(s)
char *s;
{
	if (!strncasecmp("me", s, 2) && (*(s + 2) == '\0')) {
		if (me[0] == '\0') {
			printf("  I do not know who you are.  Use IAM to tell me.\n");
			return(NULL);
		}
		else
			return(me);
	}
	else if (*s == '\0')
		return(NULL);
	else
		return(s);
}
