/* encode.c - ber output encoding routines */
/*
 * Copyright (c) 1990 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 */

#include <stdio.h>
#ifdef MACOS
#include <stdlib.h>
#include <stdarg.h>
#include "macos.h"
#else /* MACOS */
#ifdef NeXT
#include <stdlib.h>
#else /* NeXT */
#include <malloc.h>
#endif /* NeXT */
#include <varargs.h>
#include <sys/types.h>
#include <netinet/in.h>
#ifdef PCNFS
#include <tklib.h>
#endif /* PCNFS */
#endif /* MACOS */
#include <memory.h>
#include <string.h>
#include "lber.h"

#ifdef DOS
#include "msdos.h"
#endif /* DOS */

#ifdef NEEDPROTOS
static int ber_put_len( BerElement *ber, unsigned long len );
static int ber_start_seqorset( BerElement *ber, char tag );
static int ber_put_seqorset( BerElement *ber );
#endif /* NEEDPROTOS */

static ber_put_len( ber, len )
BerElement	*ber;
unsigned long	len;
{
	int		i;
	char		lenlen;
	long		mask;
	unsigned long	netlen;

	/*
	 * short len if it's less than 128 - one byte giving the len,
	 * with bit 8 0.
	 */

	if ( len <= 127 ) {
		netlen = htonl( len );
		return( ber_write( ber, (char *) &netlen + 3, 1, 0 ) );
	}

	/*
	 * long len otherwise - one byte with bit 8 set, giving the
	 * length of the length, followed by the length itself.
	 */

	/* find the first non-all-zero byte */
	for ( i = sizeof(long) - 1; i > 0; i-- ) {
		mask = (0xffL << (i * 8));
		/* not all zero */
		if ( len & mask )
			break;
	}
	lenlen = ++i;
	lenlen |= 0x80;

	/* write the length of the length */
	if ( ber_write( ber, &lenlen, 1, 0 ) != 1 )
		return( -1 );

	/* write the length itself */
	netlen = htonl( len );
	if ( ber_write( ber, (char *) &netlen + (sizeof(long) - i), i, 0 )
	    != i )
		return( -1 );

	return( i + 1 );
}

static
ber_put_int_or_enum( ber, num, tag )
BerElement	*ber;
long		num;
char		tag;
{
	int	i, sign;
	char	len;
	long	netnum, mask;

	sign = ((0x80 << ((sizeof(long) - 1) * 8)) & num);

	/*
	 * high bit is set - look for first non-all-one byte
	 * high bit is clear - look for first non-all-zero byte
	 */
	for ( i = sizeof(long) - 1; i > 0; i-- ) {
		mask = (0xffL << (i * 8));

		if ( sign ) {
			/* not all ones */
			if ( (num & mask) != mask )
				break;
		} else {
			/* not all zero */
			if ( num & mask )
				break;
		}
	}

	/*
	 * we now have the "leading byte".  if the high bit on this
	 * byte matches the sign bit, we need to "back up" a byte.
	 */
	mask = (num & (0x80L << (i * 8)));
	if ( (mask && !sign) || (sign && !mask) )
		i++;

	len = i + 1;

	if ( ber_write( ber, &tag, 1, 0 ) != 1 )
		return( -1 );

	if ( (len = ber_put_len( ber, len )) == -1 )
		return( -1 );
	i++;
	netnum = htonl( num );
	if ( ber_write( ber, (char *) &netnum + (sizeof(long) - i), i, 0 )
	   != i )
		return( -1 );

	/* length of tag + length + contents */
	return( 1 + len + i );
}

ber_put_enum( ber, num, tag )
BerElement	*ber;
long		num;
char		tag;
{
	if ( tag == LBER_DEFAULT )
		tag = LBER_ENUMERATED;

	return( ber_put_int_or_enum( ber, num, tag ) );
}

ber_put_int( ber, num, tag )
BerElement	*ber;
long		num;
char		tag;
{
	if ( tag == LBER_DEFAULT )
		tag = LBER_INTEGER;

	return( ber_put_int_or_enum( ber, num, tag ) );
}

ber_put_ostring( ber, str, len, tag )
BerElement	*ber;
char		*str;
unsigned long	len;
char		tag;
{
	int	lenlen;

	if ( tag == LBER_DEFAULT )
		tag = LBER_OCTETSTRING;

	if ( ber_write( ber, &tag, 1, 0 ) != 1 )
		return( -1 );

	if ( (lenlen = ber_put_len( ber, len )) == -1 )
		return( -1 );

	if ( ber_write( ber, str, len, 0 ) != len )
		return( -1 );

	/* return length of tag + length + contents */
	return( 1 + lenlen + len );
}

ber_put_string( ber, str, tag )
BerElement	*ber;
char		*str;
char		tag;
{
	unsigned long	len;
	int		lenlen;

	if ( tag == LBER_DEFAULT )
		tag = LBER_OCTETSTRING;

	len = strlen( str );

	if ( ber_write( ber, &tag, 1, 0 ) != 1 )
		return( -1 );

	if ( (lenlen = ber_put_len( ber, len )) == -1 )
		return( -1 );

	if ( ber_write( ber, str, len, 0 ) != len )
		return( -1 );

	/* return length of tag + length + contents */
	return( 1 + lenlen + len );
}

ber_put_bitstring( ber, str, blen, tag )
BerElement	*ber;
char		*str;
unsigned long	blen;	/* in bits */
char		tag;
{
	int		len, lenlen;
	unsigned char	unusedbits;

	if ( tag == LBER_DEFAULT )
		tag = LBER_BITSTRING;

	if ( ber_write( ber, &tag, 1, 0 ) != 1 )
		return( -1 );

	len = ( blen + 7 ) / 8;
	unusedbits = len * 8 - blen;
	if ( (lenlen = ber_put_len( ber, len + 1 )) == -1 )
		return( -1 );

	if ( ber_write( ber, (char *)&unusedbits, 1, 0 ) != 1 )
		return( -1 );

	if ( ber_write( ber, str, len, 0 ) != len )
		return( -1 );

	/* return length of tag + length + unused bit count + contents */
	return( 2 + lenlen + len );
}

ber_put_null( ber, tag )
BerElement	*ber;
char		tag;
{
	if ( tag == LBER_DEFAULT )
		tag = LBER_NULL;

	if ( ber_write( ber, &tag, 1, 0 ) != 1 )
		return( -1 );

	if ( ber_put_len( ber, 0 ) != 1 )
		return( -1 );

	return( 2 );
}

ber_put_boolean( ber, bool, tag )
BerElement	*ber;
int		bool;
char		tag;
{
	unsigned char	true = 0xff;
	unsigned char	false = 0x00;

	if ( tag == LBER_DEFAULT )
		tag = LBER_BOOLEAN;

	if ( ber_write( ber, &tag, 1, 0 ) != 1 )
		return( -1 );

	if ( ber_put_len( ber, 1 ) != 1 )
		return( -1 );

	if ( ber_write( ber, (char *)(bool ? &true : &false), 1, 0 ) != 1 )
		return( -1 );

	return( 3 );
}

#define TAG_PLUS_4BYTE_LEN	6

static ber_start_seqorset( ber, tag )
BerElement	*ber;
char		tag;
{
	Seqorset	*new;

	if ( (new = (Seqorset *) calloc( sizeof(Seqorset), 1 ))
	    == NULLSEQORSET )
		return( -1 );
	new->sos_ber = ber;
	if ( ber->ber_sos == NULLSEQORSET ) {
		new->sos_first = ber->ber_ptr;
		new->sos_ptr = ber->ber_ptr + TAG_PLUS_4BYTE_LEN;
	} else {
		new->sos_first = ber->ber_sos->sos_ptr;
		new->sos_ptr = ber->ber_sos->sos_ptr + TAG_PLUS_4BYTE_LEN;
	}
	new->sos_tag = tag;

	new->sos_next = ber->ber_sos;
	ber->ber_sos = new;

	return( 0 );
}

ber_start_seq( ber, tag )
BerElement	*ber;
char		tag;
{
	if ( tag == LBER_DEFAULT )
		tag = LBER_SEQUENCE;

	return( ber_start_seqorset( ber, tag ) );
}

ber_start_set( ber, tag )
BerElement	*ber;
char		tag;
{
	if ( tag == LBER_DEFAULT )
		tag = LBER_SET;

	return( ber_start_seqorset( ber, tag ) );
}

static ber_put_seqorset( ber )
BerElement	*ber;
{
	unsigned long	len, netlen;
	unsigned char	ltag = 0x80 + sizeof(unsigned long);
	Seqorset	*next;
	Seqorset	**sos = &ber->ber_sos;

	/*
	 * If this is the toplevel sequence or set, we need to actually
	 * write the stuff out.  Otherwise, it's already been put in
	 * the appropriate buffer and will be written when the toplevel
	 * one is written.  In this case all we need to do is update the
	 * length and tag.
	 */

	len = (*sos)->sos_clen;
	netlen = htonl( len );
	if ( (next = (*sos)->sos_next) == NULLSEQORSET ) {
		/* one byte tag */
		if ( ber_write( ber, (char *)&((*sos)->sos_tag), 1, 1 ) != 1 )
			return( -1 );

		/* one byte of length length */
		if ( ber_write( ber, (char *)&ltag, 1, 1 ) != 1 )
			return( -1 );

		/* the length itself */
		if ( ber_write( ber, (char *) &netlen, sizeof(unsigned long),
		    1 ) != sizeof(unsigned long) )
			return( -1 );

		(*sos)->sos_ber->ber_ptr += len;
	} else {
		/* one byte tag */
		(void) memcpy( (*sos)->sos_first, &((*sos)->sos_tag), 1 );

		/* one byte of length length */
		(void) memcpy( (*sos)->sos_first + 1, &ltag, 1 );

		/* the length itself */
		(void) memcpy( (*sos)->sos_first + 2, &netlen,
		    sizeof(unsigned long) );

		next->sos_clen += (TAG_PLUS_4BYTE_LEN + len);
		next->sos_ptr += (TAG_PLUS_4BYTE_LEN + len);
	}

	/* we're done with this seqorset, so free it up */
	free( (char *) (*sos) );
	*sos = next;

	return( TAG_PLUS_4BYTE_LEN + len );
}

ber_put_seq( ber )
BerElement	*ber;
{
	return( ber_put_seqorset( ber ) );
}

ber_put_set( ber )
BerElement	*ber;
{
	return( ber_put_seqorset( ber ) );
}

/* VARARGS */
ber_printf(
#ifdef MACOS
	BerElement *ber, char *fmt, ... )
#else /* MACOS */
	va_alist )
va_dcl
#endif /* MACOS */
{
	va_list		ap;
#ifndef MACOS
	BerElement	*ber;
	char		*fmt;
#endif /* MACOS */
	char		*s, **ss;
	struct berval	**bv;
	int		rc, i;
	unsigned long	len;

#ifdef MACOS
	va_start( ap, fmt );
#else /* MACOS */
	va_start( ap );
	ber = va_arg( ap, BerElement * );
	fmt = va_arg( ap, char * );
#endif /* MACOS */

	for ( rc = 0; *fmt && rc != -1; fmt++ ) {
		switch ( *fmt ) {
		case 'b':	/* boolean */
			i = va_arg( ap, int );
			rc = ber_put_boolean( ber, i, (char)ber->ber_tag );
			break;

		case 'i':	/* int */
			i = va_arg( ap, int );
			rc = ber_put_int( ber, i, (char)ber->ber_tag );
			break;

		case 'e':	/* enumeration */
			i = va_arg( ap, int );
			rc = ber_put_enum( ber, i, (char)ber->ber_tag );
			break;

		case 'n':	/* null */
			rc = ber_put_null( ber, (char)ber->ber_tag );
			break;

		case 'o':	/* octet string (non-null terminated) */
			s = va_arg( ap, char * );
			len = va_arg( ap, int );
			rc = ber_put_ostring( ber, s, len, (char)ber->ber_tag );
			break;

		case 's':	/* string */
			s = va_arg( ap, char * );
			rc = ber_put_string( ber, s, (char)ber->ber_tag );
			break;

		case 'B':	/* bit string */
			s = va_arg( ap, char * );
			len = va_arg( ap, int );	/* in bits */
			rc = ber_put_bitstring( ber, s, len, (char)ber->ber_tag );
			break;

		case 't':	/* tag for the next element */
			ber->ber_tag = va_arg( ap, int );
			ber->ber_usertag = 1;
			break;

		case 'v':	/* vector of strings */
			if ( (ss = va_arg( ap, char ** )) == NULL )
				break;
			for ( i = 0; ss[i] != NULL; i++ ) {
				if ( (rc = ber_put_string( ber, ss[i],
				    (char)ber->ber_tag )) == -1 )
					break;
			}
			break;

		case 'V':	/* sequences of strings + lengths */
			if ( (bv = va_arg( ap, struct berval ** )) == NULL )
				break;
			for ( i = 0; bv[i] != NULL; i++ ) {
				if ( (rc = ber_put_ostring( ber, bv[i]->bv_val,
				    bv[i]->bv_len, (char)ber->ber_tag )) == -1 )
					break;
			}
			break;

		case '{':	/* begin sequence */
			rc = ber_start_seq( ber, (char)ber->ber_tag );
			break;

		case '}':	/* end sequence */
			rc = ber_put_seqorset( ber );
			break;

		case '[':	/* begin set */
			rc = ber_start_set( ber, (char)ber->ber_tag );
			break;

		case ']':	/* end set */
			rc = ber_put_seqorset( ber );
			break;

		default:
#ifndef NO_USERINTERFACE
			fprintf( stderr, "unknown fmt %c\n", *fmt );
#endif /* NO_USERINTERFACE */
			rc = -1;
			break;
		}

		if ( ber->ber_usertag == 0 )
			ber->ber_tag = LBER_DEFAULT;
		else
			ber->ber_usertag = 0;
	}

	va_end( ap );

	return( rc );
}
