/*
 * Copyright (c) 1990 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <quipu/commonarg.h>
#include <quipu/attrvalue.h>
#include <quipu/ds_error.h>
#include <quipu/ds_search.h>
#include <quipu/dap2.h>
#include <quipu/dua.h>
#include "lber.h"
#include "ldap.h"

short	ldap_photo_syntax;
short	ldap_jpeg_syntax;
short	ldap_audio_syntax;
short	ldap_dn_syntax;
short	ldap_postaladdress_syntax;
short	ldap_acl_syntax;
short	ldap_mtai_syntax;
short	ldap_rts_cred_syntax;
short	ldap_rtl_syntax;

AttributeValue	ldap_str2AttrV();
static		de_t61();

static get_one_syntax( attrib, required )
char	*attrib;
int	required;
{
	oid_table_attr	*p, *name2attr();

	if ( (p = name2attr( attrib )) != (oid_table_attr *) 0 )
	    return( p->oa_syntax );

	if ( !required )
	    return( -1 );

	Debug( LDAP_DEBUG_ANY, "name2attr (%s) failed - exiting\n", attrib,
	    0, 0 );

	log_and_exit( 1 );
}

get_syntaxes()
{
	oid_table_attr	*name2attr();

	Debug( LDAP_DEBUG_TRACE, "get_syntaxes\n", 0, 0, 0 );

	ldap_photo_syntax = get_one_syntax( "photo", 0 );
	ldap_jpeg_syntax = get_one_syntax( "jpegPhoto", 0 );
	ldap_audio_syntax = get_one_syntax( "audio", 0 );
	ldap_postaladdress_syntax = get_one_syntax( "postaladdress", 0 );
	ldap_dn_syntax = get_one_syntax( "aliasedObjectName", 1 );
	ldap_acl_syntax = get_one_syntax( "acl", 0 );
	ldap_mtai_syntax = get_one_syntax( "mTAInfo", 0 );
	ldap_rts_cred_syntax= get_one_syntax( "initiatingRTSCredentials", 0 );
	ldap_rtl_syntax= get_one_syntax( "routingTreeList", 0 );
}

#define SEPARATOR(c)	(c == ',' || c == ';')
#define SPACE(c)	(c == ' ' || c == '\n')

dn_print_real( ps, dn, format )
PS	ps;
DN	dn;
int	format;
{
	RDN	rdn;
	int	firstrdn;
	char	*value;
	PS	rps;
	void	ldap_dn_print();

	if ( dn == NULLDN )
		return( 0 );

	if ( dn->dn_parent != NULLDN ) {
		dn_print_real( ps, dn->dn_parent, format );
		ps_print( ps, ", " );
	}

	if ( (rps = ps_alloc( str_open )) == NULLPS )
		return( -1 );
	if ( str_setup( rps, NULLCP, 0, 0 ) == NOTOK )
		return( -1 );

	firstrdn = 1;
	for ( rdn = dn->dn_rdn; rdn != NULLRDN; rdn = rdn->rdn_next ) {
		if ( firstrdn )
			firstrdn = 0;
		else
			ps_print( ps, " + " );

		AttrT_print( ps, rdn->rdn_at, format );
		ps_print( ps, "=" );

		if ( rdn->rdn_at->oa_syntax == ldap_dn_syntax ) {
			dn_print_real( rps, (DN) &rdn->rdn_av.av_struct,
			    format );
			*rps->ps_ptr = '\0';
			value = rps->ps_base;
		} else {
			AttrV_print( rps, &rdn->rdn_av, EDBOUT );
			*rps->ps_ptr = '\0';
			value = rps->ps_base;
			de_t61( value, 0 );
		}

		/*
		 * ,+="\\\n all go in quotes.  " and \\ need to
		 * be preceeded by \\.
		 */

		if ( strpbrk( value, ",+=\"\\\n" ) != NULL || SPACE( value[0] )
		    || SPACE( value[max( strlen(value) - 1, 0 )] ) ) {
			char	*p, *t, *tmp;
			int	specialcount;

			ps_print( ps, "\"" );

			specialcount = 0;
			for ( p = value; *p != '\0'; p++ ) {
				if ( *p == '"' || *p == '\\' ) {
					specialcount++;
				}
			}
			if ( specialcount > 0 ) {
				tmp = smalloc( strlen( value ) + specialcount
				    + 1 );
				for ( p = value, t = tmp; *p != '\0'; p++ ) {
					switch ( *p ) {
					case '"':
					case '\\':
						*t++ = '\\';
						/* FALL THROUGH */
					default:
						*t++ = *p;
					}
				}
				*t = '\0';
				ps_print( ps, tmp );
				free( tmp );
			} else {
				ps_print( ps, value );
			}

			ps_print( ps, "\"" );
		} else {
			ps_print( ps, value );
		}

		rps->ps_ptr = rps->ps_base;
	}

	ps_free( rps );

	return( 0 );
}

void ldap_dn_print( ps, dn, format )
PS	ps;
DN	dn;
int	format;
{
	Debug( LDAP_DEBUG_TRACE, "ldap_dn_print\n", 0, 0, 0 );

	dn_print_real( ps, dn, format );
}

encode_dn( ber, dn )
BerElement	*ber;
DN		dn;
{
	PS	ps;
	int	rc;

	Debug( LDAP_DEBUG_TRACE, "encode_dn\n", 0, 0, 0 );

	if ( (ps = ps_alloc( str_open )) == NULLPS )
		return( -1 );
	if ( str_setup( ps, NULLCP, 0, 0 ) == NOTOK )
		return( -1 );

	ldap_dn_print( ps, dn, EDBOUT );
	*ps->ps_ptr = '\0';

	rc = ber_printf( ber, "s", ps->ps_base );

	ps_free( ps );

	return( rc );
}

static put_jpeg_value( ber, av )
BerElement	*ber;
AttributeValue	av;
{
	PE	pe;
	int	len;

	Debug( LDAP_DEBUG_TRACE, "put_jpeg_value\n", 0, 0, 0 );

	pe = (PE) (((struct file_syntax *) av->av_struct)->fs_attr->av_struct);

	if ( (pe->pe_class != PE_CLASS_UNIV && pe->pe_class != PE_CLASS_CONT)
	    || pe->pe_form != PE_FORM_PRIM || pe->pe_id != PE_PRIM_OCTS ) {
		Debug( LDAP_DEBUG_ANY, "put_jpeg_value: unkonwn type\n", 0,
		    0, 0 );
		return( -1 );
	}

	if ( pe_pullup( pe ) == NOTOK ) {
		Debug( LDAP_DEBUG_ANY, "put_jpeg_value: cannot pullup\n", 0,
		    0, 0 );
		return( -1 );
	}

	len = ps_get_abs( pe );

	Debug( LDAP_DEBUG_ARGS, "put_jeg_value: ber_printf %d bytes\n",
	    len, 0, 0 );
	if ( ber_printf( ber, "o", (char *) pe->pe_prim, len ) == -1 ) {
		Debug( LDAP_DEBUG_ANY, "put_jpeg_value: ber_printf failed\n",
		    0, 0, 0 );
		return( -1 );
	}

	return( 0 );
}

static put_audio_value( ber, av )
BerElement	*ber;
AttributeValue	av;
{
	struct qbuf	*qb, *p;
	int		rc, len;
	char		*buf;

	Debug( LDAP_DEBUG_TRACE, "put_audio_value\n", 0, 0, 0 );

	qb = (struct qbuf *) (((struct file_syntax *)
	    av->av_struct)->fs_attr->av_struct);

	len = 0;
	for ( p = qb->qb_forw; p != qb; p = p->qb_forw ) {
		len += p->qb_len;
	}

	if ( (buf = (char *) malloc( len )) == NULL )
		return( -1 );

	len = 0;
	for ( p = qb->qb_forw; p != qb; p = p->qb_forw ) {
		memcpy( buf + len, p->qb_data, p->qb_len );
		len += p->qb_len;
	}

	Debug( LDAP_DEBUG_ARGS, "put_audio_value: ber_printf %d bytes\n",
	    len, 0, 0 );

	if ( (rc = ber_printf( ber, "o", buf, len )) == -1 )
		Debug( LDAP_DEBUG_ANY, "put_audio_value: ber_printf failed\n",
		    0, 0, 0 );

	free( buf );

	return( rc );
}

static put_photo_value( ber, av )
BerElement	*ber;
AttributeValue	av;
{
	PE		pe;
	PS		ps;
	int		len;
	char		*faxparamset = "\000\300\000\000";
	BerElement	*phber;

	Debug( LDAP_DEBUG_TRACE, "put_photo_value\n", 0, 0, 0 );

	pe = (PE) (((struct file_syntax *) av->av_struct)->fs_attr->av_struct);

	/* old bit string-like format - only handle this for now */
	if ( pe->pe_class == PE_CLASS_UNIV && pe->pe_form == PE_FORM_PRIM
	    && pe->pe_id == PE_PRIM_BITS ) {
		len = ps_get_abs( pe );
		Debug( LDAP_DEBUG_ARGS, "put_photo_val: ber_printf %d bytes\n",
		    len, 0, 0 );
		if (( phber = ber_alloc()) == NULLBER ) {
			Debug( LDAP_DEBUG_ANY, "ber_alloc failed\n", 0, 0, 0 );
			return( -1 );
		}
		if ( ber_printf( phber, "t{[tB]{B}}", 0xA3, 0x81, faxparamset,
		    31, (char *)pe->pe_prim, len * 8 ) == -1 ) {
			Debug( LDAP_DEBUG_ANY, "ber_printf failed\n", 0, 0, 0 );
			ber_free( phber, 1 );
			return( -1 );
		}
		if ( ber_printf( ber, "o", phber->ber_buf, phber->ber_ptr
		    - phber->ber_buf ) == -1 ) {
			Debug( LDAP_DEBUG_ANY, "ber_printf failed\n", 0, 0, 0 );
			ber_free( phber, 1 );
			return( -1 );
		}
		ber_free( phber, 1 );
	} else {
		/*
		 * try just writing this into a PS and sending it along
		 */
		ps_len_strategy = PS_LEN_LONG;
		if ( (ps = ps_alloc( str_open )) == NULLPS )
			return( -1 );
		if ( str_setup( ps, NULLCP, 0, 0 ) == NOTOK ||
		    pe2ps( ps, pe ) == NOTOK ) {
			ps_free( ps );
			return( -1 );
		}

		len = ps->ps_ptr - ps->ps_base;
		Debug( LDAP_DEBUG_ARGS, "put_photo_val: ber_printf %d bytes\n",
		    len, 0, 0 );
		if ( ber_printf( ber, "o", (char *) ps->ps_base, len ) == -1 ) {
			Debug( LDAP_DEBUG_ANY, "ber_printf failed\n", 0, 0, 0 );
			ps_free( ps );
			return( -1 );
		}
		ps_free( ps );
	}

	return( 0 );
}

static put_values( ber, ps, syntax, vals )
BerElement	*ber;
PS		ps;
short		syntax;
AV_Sequence	vals;
{
	AV_Sequence	av;

	Debug( LDAP_DEBUG_TRACE, "put_values\n", 0, 0, 0 );

	for ( av = vals; av != NULLAV; av = av->avseq_next ) {
		if ( syntax == ldap_jpeg_syntax ) {
			if ( put_jpeg_value( ber, &av->avseq_av ) == -1 )
				return( -1 );
		} else if ( syntax == ldap_photo_syntax ) {
			if ( put_photo_value( ber, &av->avseq_av ) == -1 )
				return( -1 );
		} else if ( syntax == ldap_audio_syntax ) {
			if ( put_audio_value( ber, &av->avseq_av ) == -1 )
				return( -1 );
		} else if ( syntax == ldap_dn_syntax ) {
			if ( encode_dn( ber, av->avseq_av.av_struct ) == -1 )
				return( -1 );
		} else if ( syntax > AV_WRITE_FILE ) {
			struct file_syntax	*fsyntax;

			fsyntax = (struct file_syntax *) av->avseq_av.av_struct;

			ps->ps_ptr = ps->ps_base;
			AttrV_print( ps, fsyntax->fs_attr, EDBOUT );
			*ps->ps_ptr = '\0';

			if ( ber_printf( ber, "o", ps->ps_base,
			    ps->ps_ptr - ps->ps_base ) == -1 )
				return( -1 );
		} else {
			ps->ps_ptr = ps->ps_base;
			AttrV_print( ps, &av->avseq_av, EDBOUT );
			*ps->ps_ptr = '\0';
			de_t61( ps->ps_base, 0 );

			if ( ber_printf( ber, "s", ps->ps_base ) == -1 )
				return( -1 );
		}
	}

	return( 0 );
}

encode_attrs( ber, as )
BerElement	*ber;
Attr_Sequence	as;
{
	PS		ps;
#ifdef OLDBROKEN
	extern int	oldbroken;
#endif

	Debug( LDAP_DEBUG_TRACE, "encode_attrs\n", 0, 0, 0 );

	if ( (ps = ps_alloc( str_open )) == NULLPS )
		return( -1 );
	if ( str_setup( ps, NULLCP, 0, 0 ) == NOTOK )
		return( -1 );

#ifdef OLDBROKEN
	if ( ber_printf( ber, "t{", oldbroken ? OLD_LBER_SEQUENCE :
	    LBER_SEQUENCE ) == -1 ) {
#else
	if ( ber_printf( ber, "{" ) == -1 ) {
#endif
		ps_free( ps );
		return( -1 );
	}

	while ( as != NULLATTR ) {
		ps->ps_ptr = ps->ps_base;
		AttrT_print( ps, as->attr_type, EDBOUT );
		*ps->ps_ptr = '\0';

#ifdef OLDBROKEN
		if ( ber_printf( ber, "t{st[", oldbroken ? OLD_LBER_SEQUENCE :
		    LBER_SEQUENCE, ps->ps_base, oldbroken ? OLD_LBER_SET :
		    LBER_SET ) == -1 ) {
#else
		if ( ber_printf( ber, "{s[", ps->ps_base ) == -1 ) {
#endif
			ps_free( ps );
			return( -1 );
		}

		put_values( ber, ps, as->attr_type->oa_syntax, as->attr_value );

		if ( ber_printf( ber, "]}" ) == -1 ) {
			ps_free( ps );
			return( -1 );
		}

		as = as->attr_link;
	}
	ps_free( ps );

	if ( ber_printf( ber, "}" ) == -1 )
		return( -1 );

	return( 0 );
}

DN ldap_str2dn( str )
char	*str;
{
	DN		dn, save;
	RDN		rdn, newrdn, tmprdn;
	AttributeType	at;
	AttributeValue	av;
	char		*type, *value, *savestr;
	int		morerdncomps;

	Debug( LDAP_DEBUG_TRACE, "ldap_str2dn\n", 0, 0, 0 );

	savestr = str = strdup( str );
	dn = NULLDN;
	do {
		char	*r;
		int	state;

		rdn = NULLRDN;
		morerdncomps = 1;
		do {
			/* get the type */
			while ( *str == ' ' || *str == '\n' )
				str++;
			type = str;
			while ( *str != '\0' && *str != '=' )
				str++;
			if ( *str == '\0' ) {
				free( savestr );
				Debug( LDAP_DEBUG_ARGS, "no =\n", 0, 0, 0 );
				return( NULLDN );
			}
			*str++ = '\0';
			if ( strncmp( type, "OID.", 4 ) == 0 )
				type += 4;

#define BEGINVALUE	1
#define INVALUE		2
#define INQUOTE 	3
#define ENDVALUE	4
			r = value = str;
			state = BEGINVALUE;
			/* break or return out */
			while ( state != ENDVALUE ) {
				switch ( *str ) {
				case '"':
					if ( state == BEGINVALUE ) {
						state = INQUOTE;
						str++;
					} else if ( state == INQUOTE ) {
						state = ENDVALUE;
						str++;
					} else {
						free( savestr );
						Debug( LDAP_DEBUG_ARGS,
						    "quote state %d\n", state,
						    0, 0 );
						return( NULLDN );
					}
					break;

				case ',':
				case ';':
				case '+':
					if ( state == INVALUE ) {
						state = ENDVALUE;
					} else if ( state == INQUOTE ) {
						*r++ = *str++;
					} else {
						free( savestr );
						Debug( LDAP_DEBUG_ARGS,
						    "comma state %d\n", state,
						    0, 0 );
						return( NULLDN );
					}
					break;

				case ' ':
				case '\n':
					if ( state == BEGINVALUE ) {
						str++;
					} else {
						*r++ = *str++;
					}
					break;

				case '\\':
					str++;
					*r++ = *str++;
					break;

				case '\0':
					state = ENDVALUE;
					break;

				default:
					if ( state == BEGINVALUE )
						state = INVALUE;
					*r++ = *str++;
					break;
				}
			}

			while ( SPACE( *str ) )
				str++;
			if ( *str == '+' ) {
				morerdncomps = 1;
				str++;
			} else {
				morerdncomps = 0;
				if ( SEPARATOR( *str ) )
					str++;
			}
			*r = '\0';

			/* type */
			if ( (at = str2AttrT( type )) == NULLAttrT ) {
				dn_free( dn );
				free( savestr );
				Debug( LDAP_DEBUG_ARGS, "bad type (%s)\n",
				    type, 0, 0 );
				return( NULLDN ); /* LDAP_UNDEFINED_TYPE */
			}
			/* value */
			if ( (av = ldap_str2AttrV( value, at->oa_syntax ))
			    == NULLAttrV ) {
				dn_free( dn );
				free( savestr );
				Debug( LDAP_DEBUG_ARGS, "bad val\n", 0, 0, 0 );
				return( NULLDN ); /* LDAP_INVALID_SYNTAX */
			}
			/* make the rdn */
			newrdn = rdn_comp_new( at, av );

			/* add it to the list */
			for ( tmprdn = rdn; tmprdn != NULLRDN &&
			    tmprdn->rdn_next != NULLRDN;
			    tmprdn = tmprdn->rdn_next )
				;	/* NULL */
			if ( tmprdn != NULLRDN )
				tmprdn->rdn_next = newrdn;
			else
				rdn = newrdn;

			AttrV_free( av );
		} while ( morerdncomps );

		save = dn;
		dn = dn_comp_new( rdn );
		dn->dn_parent = save;
	} while ( str != NULL && *str != '\0' );

	free( savestr );
	Debug( LDAP_DEBUG_TRACE, "ldap_str2dn OK\n", 0, 0, 0 );
	return( dn );
}

#define T61	"{T.61}"
#define T61LEN	6

static de_t61( s, t61mark )
char	*s;
int	t61mark;
{
	char	*next = s;
	int	c, hex;

	while ( *s ) {
		switch ( *s ) {
		case '{' :
			if ( strncasecmp( s, T61, T61LEN) == 0 ) {
				s += T61LEN;
				if ( t61mark )
					*next++ = '@';
			} else {
				*next++ = *s++;
			}
			break;

		case '\\':
			c = *(s + 1);
			if ( c == '\n' ) {
				s += 2;
				if ( *s == '\t' )
					s++;
				break;
			}
			if ( isdigit( c ) )
				hex = c - '0';
			else if ( c >= 'A' && c <= 'F' )
				hex = c - 'A' + 10;
			else if ( c >= 'a' && c <= 'f' )
				hex = c - 'a' + 10;
			else {
				*next++ = *s++;
				break;
			}
			hex <<= 4;
			c = *(s + 2);
			if ( isdigit( c ) )
				hex += c - '0';
			else if ( c >= 'A' && c <= 'F' )
				hex += c - 'A' + 10;
			else if ( c >= 'a' && c <= 'f' )
				hex += c - 'a' + 10;
			else {
				*next++ = *s++;
				*next++ = *s++;
				break;
			}

			*next++ = hex;
			s += 3;
			break;

		default:
			*next++ = *s++;
			break;
		}
	}
	*next = '\0';
}


PE
bv_asn2pe( bv )
	struct berval	*bv;
{
	PS	ps;
	PE	pe;

	if (( ps = ps_alloc(str_open)) == NULLPS || str_setup( ps, bv->bv_val,
	    bv->bv_len, 0 ) == NOTOK ) {
		Debug( LDAP_DEBUG_TRACE, "bv_asn2pe: ps_alloc failed\n",
		    0, 0, 0 );
		return( NULLPE );
	}

	pe = ps2pe( ps );
	if ( ps->ps_errno != PS_ERR_NONE ) {
		Debug( LDAP_DEBUG_TRACE, "bv_asn2pe: ps2pe failed %s\n",
		    ps_error(ps->ps_errno), 0, 0 );
		if ( pe != NULLPE ) {
			pe_free( pe );
		}
		return( NULLPE );
	}

	return( pe );
}


AttributeValue
bv_octet2AttrV( bv )
	struct berval	*bv;
{
	AttributeValue	av;

	av = AttrV_alloc();
	if ( av == NULLAttrV ) {
		return( NULLAttrV );
	}

	if (( av->av_struct = (caddr_t) str2prim( bv->bv_val, bv->bv_len,
	    PE_CLASS_UNIV, PE_PRIM_OCTS )) == NULL ) {
		free((char *)av );
		return( NULLAttrV );
	}

	av->av_syntax = 0;
	return( av );
}


AttributeValue
bv_asn2AttrV( bv )
	struct berval	*bv;
{
	AttributeValue	av;

	av = AttrV_alloc();
	if ( av == NULLAttrV ) {
		return( NULLAttrV );
	}

	if (( av->av_struct = (caddr_t) bv_asn2pe( bv )) == NULL ) {
		free((char *)av );
		return( NULLAttrV );
	}

	av->av_syntax = 0;
	return( av );
}


AttributeValue
ldap_strdn2AttrV( dnstr )
    char	*dnstr;
{
	DN		dn;
	AttributeValue	av;

	if (( dn = ldap_str2dn( dnstr )) == NULL ) {
		return( NULLAttrV );
	}

	av = AttrV_alloc();
	if ( av == NULLAttrV ) {
		dn_free( dn );
		return( NULLAttrV );
	}

	av->av_struct = (caddr_t)dn; 
	av->av_syntax = ldap_dn_syntax;
	return( av );
}

RDN
ldap_str2rdn( rdnstr )
    char	*rdnstr;
{
	DN	dn;
	RDN	rdn;

	if ( (dn = ldap_str2dn( rdnstr )) == NULL ) {
		return( NULL );
	}

	if ( (rdn = rdn_cpy( dn->dn_rdn )) == NULL ) {
		return( NULL );
	}

	dn_free( dn );

	return( rdn );
}

AttributeValue
ldap_str_at2AttrV( str, type )
    char		*str;
    AttributeType	type;
{
	char		*s, *res, *r;
	AttributeValue	str_at2AttrV();

	Debug( LDAP_DEBUG_TRACE, "ldap_str_at2AttrV str (%s) type (%s)\n", str,
	    type->oa_ot.ot_name, 0 );

	if ( type->oa_syntax == ldap_rts_cred_syntax ||
	    type->oa_syntax == ldap_mtai_syntax ||
	    type->oa_syntax == ldap_rtl_syntax ) {
		res = str;
	} else {
		res = (char *) malloc( max( 2 * strlen( str ), 10 ) );

		r = res;
		for ( s = str; *s; s++ ) {
			switch ( *s ) {
			case '&':
			case '#':
			case '$':
			case '%':
			case '@':
				sprintf( r, "\\%02x", *s & 0xff );
				r += 3;
				break;

			default:
				*r++ = *s;
			}
		}
		*r = '\0';
	}

	Debug( LDAP_DEBUG_TRACE, "ldap_str_at2AttrV returning (%s)\n", res,
	    0, 0 );

	return( str_at2AttrV( res, type ) );
}

AttributeValue
ldap_str2AttrV( value, syntax )
    char	*value;
    short	syntax;
{
	if ( syntax == ldap_dn_syntax ) {
		return( ldap_strdn2AttrV( value ) );
	} else {
		return( str2AttrV( value, syntax ) );
	}
}
