/*
 * Copyright (c) 1990 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 */

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/wait.h>
#include <signal.h>
#include <quipu/commonarg.h>
#include <quipu/ds_error.h>
#include <quipu/dap2.h>
#include <quipu/dua.h>
#ifdef __hpux
#include <syslog.h>
#else
#include <sys/syslog.h>
#endif
#include "lber.h"
#include "ldap.h"
#include "common.h"

/*
 * client_request - called by do_queries() when there is activity on the
 * client socket.  It expects to be able to get an LDAP message from the
 * client socket, parses the first couple of fields, and then calls
 * do_request() to handle the request.  If do_request() (or something
 * called by it) returns a response to the client (e.g., in the case of
 * an error), then client_request() is done.  If the request is not
 * responded to (and needs a response), it is added to the queue of
 * outstanding requests.  It will be responded to later via dsa_response(),
 * once the DSA operation completes.
 */

client_request( clientsb, dsaconn )
Sockbuf		*clientsb;
struct conn	*dsaconn;
{
	int		tag;
	int		len;
	int		msgid;
	BerElement	ber, *copyofber;
	struct msg	*m;
	static int	bound;
	extern char	*bound_dn, *bound_pw;
	struct PSAPaddr	*bound_addr, *psap_cpy();
#ifdef OLDBROKEN
	extern int	oldbroken;
#endif

	Debug( LDAP_DEBUG_TRACE, "client_request\n", 0, 0, 0 );

	/*
	 * Get the ldap message, which is a sequence of message id
	 * and then the actual request choice.
	 */

	if ( (tag = ber_get_next( clientsb, &len, &ber )) == -1 ) {
		Debug( LDAP_DEBUG_ANY, "ber_get_next failed\n", 0, 0, 0 );
		log_and_exit( 1 );
	}

#ifdef LDAP_DEBUG
	if ( ldap_debug & LDAP_DEBUG_BER )
		trace_ber( tag, len, ber.ber_buf, stderr, 1 );
#endif

#ifdef OLDBROKEN
	/*
	 * This tag should be a normal SEQUENCE tag.  In the old broken
	 * stuff this is 0x10.  In the new stuff this is 0x30.  We decide
	 * here which version we're dealing with.
	 */

	if ( tag == OLD_LDAP_TAG_MESSAGE && oldbroken == 0 ) {
		Debug( LDAP_DEBUG_ANY, "OLD version detected tag %d\n", tag,
		    0, 0 );
		syslog( LOG_INFO, "old broken version detected" );
		oldbroken = 1;
	}
#endif

	if ( ber_get_int( &ber, &msgid ) != LDAP_TAG_MSGID ) {
		send_ldap_result( clientsb, LBER_DEFAULT, msgid,
		    LDAP_PROTOCOL_ERROR, "Not an LDAP message" );
		return;
	}

#ifdef OLDBROKEN
	if ( oldbroken )
		tag = ber_peek_tag( &ber, &len );
	else
#endif
	tag = ber_skip_tag( &ber, &len );
	if ( bound == 0 && tag != LDAP_REQ_BIND
#ifdef OLDBROKEN
	    && tag != OLD_LDAP_REQ_BIND
#endif
	    ) {
		send_ldap_result( clientsb, tag, msgid, LDAP_OPERATIONS_ERROR,
		    "Bind operation must come first" );
		return;
	}

	copyofber = ber_dup( &ber );

	m = add_msg( msgid, tag, copyofber, dsaconn );

	/* 
	 * Call the appropriate routine to handle the request.  If it
	 * returns a nonzero result, the message requires a response, and
	 * so it's left in the queue of outstanding requests, otherwise
	 * it's deleted.
	 */

	if ( do_request( clientsb, m, &ber, &bound ) == 0 ) {
		del_msg( msgid );
	}

	return;
}

/*
 * do_request - called when a client makes a request, or when a referral
 * error is returned.  In the latter case, a connection is made to the
 * referred to DSA, and do_request() is called to retry the operation over
 * that connection.  In the former case, do_request() is called to try
 * the operation over the default association.
 */

do_request( clientsb, m, ber, bound )
Sockbuf		*clientsb;
struct msg	*m;
BerElement	*ber;
int		*bound;
{
	int		resp_required = 0;

	Debug( LDAP_DEBUG_TRACE, "do_request\n", 0, 0, 0 );

	switch ( m->m_msgtype ) {
#ifdef OLDBROKEN
	case OLD_LDAP_REQ_BIND:
#endif
	case LDAP_REQ_BIND:
		resp_required = do_bind( clientsb, m, ber, bound );
		break;

#ifdef OLDBROKEN
	case OLD_LDAP_REQ_UNBIND:
#endif
	case LDAP_REQ_UNBIND:
		conn_close();
		log_and_exit( 0 );
		break;

#ifdef OLDBROKEN
	case OLD_LDAP_REQ_ADD:
#endif
	case LDAP_REQ_ADD:
		resp_required = do_add( clientsb, m, ber );
		break;

#ifdef OLDBROKEN
	case OLD_LDAP_REQ_DELETE:
#endif
	case LDAP_REQ_DELETE:
		resp_required = do_delete( clientsb, m, ber );
		break;

#ifdef OLDBROKEN
	case OLD_LDAP_REQ_MODRDN:
#endif
	case LDAP_REQ_MODRDN:
		resp_required = do_modrdn( clientsb, m, ber );
		break;

#ifdef OLDBROKEN
	case OLD_LDAP_REQ_MODIFY:
#endif
	case LDAP_REQ_MODIFY:
		resp_required = do_modify( clientsb, m, ber );
		break;

#ifdef OLDBROKEN
	case OLD_LDAP_REQ_COMPARE:
#endif
	case LDAP_REQ_COMPARE:
		resp_required = do_compare( clientsb, m, ber );
		break;

#ifdef OLDBROKEN
	case OLD_LDAP_REQ_SEARCH:
#endif
	case LDAP_REQ_SEARCH:
		resp_required = do_search( clientsb, m, ber );
		break;

#ifdef OLDBROKEN
	case OLD_LDAP_REQ_ABANDON:
#endif
	case LDAP_REQ_ABANDON:
		resp_required = do_abandon( m->m_conn, ber, m->m_msgid );
		break;

	default:
		Debug( LDAP_DEBUG_ANY, "unknown operation %d\n", m->m_msgtype,
		    0, 0 );

		send_ldap_result( clientsb, m->m_msgtype, m->m_msgid,
		    LDAP_PROTOCOL_ERROR, "Unknown request type" );
		break;
	}

	return( resp_required );
}

/* 
 * initiate_dap_operation - initiate a dap operation, rebinding and retrying
 * the request if necessary.  If the request is successfully initiated, 0 is
 * returned.  Otherwise, an indication of the error is returned.
 */

initiate_dap_operation( op, m, arg )
int		op;
struct msg	*m;
caddr_t		arg;
{
	char			*matched;
	int			i, rc, bound = 0;
	struct DAPindication	di;

	Debug( LDAP_DEBUG_TRACE, "initiate_dap_operation\n", 0, 0, 0 );

	if ( m->m_conn->c_ad == -1 && do_bind_real( m->m_conn, &bound,
	    &matched ) != LDAP_SUCCESS )
		return( LDAP_UNAVAILABLE );

	for ( i = 0; i < 2; i++ ) {
		switch ( op ) {
		case OP_COMPARE:
			rc = DapCompare( m->m_conn->c_ad, m->m_msgid,
			    (struct ds_compare_arg *) arg, &di, ROS_ASYNC );
			break;

		case OP_SEARCH:
			rc = DapSearch( m->m_conn->c_ad, m->m_msgid,
			    (struct ds_search_arg *) arg, &di, ROS_ASYNC );
			break;

		case OP_ADDENTRY:
			rc = DapAddEntry( m->m_conn->c_ad, m->m_msgid,
			    (struct ds_addentry_arg *) arg, &di, ROS_ASYNC );
			break;

		case OP_REMOVEENTRY:
			rc = DapRemoveEntry( m->m_conn->c_ad, m->m_msgid,
			    (struct ds_removeentry_arg *) arg, &di, ROS_ASYNC );
			break;

		case OP_MODIFYENTRY:
			rc = DapModifyEntry( m->m_conn->c_ad, m->m_msgid,
			    (struct ds_modifyentry_arg *) arg, &di, ROS_ASYNC );
			break;

		case OP_READ:
			rc = DapRead( m->m_conn->c_ad, m->m_msgid,
			    (struct ds_read_arg *) arg, &di, ROS_ASYNC );
			break;

		case OP_MODIFYRDN:
			rc = DapModifyRDN( m->m_conn->c_ad, m->m_msgid,
			    (struct ds_modifyrdn_arg *) arg, &di, ROS_ASYNC );
			break;

		default:
			break;
		}

		Debug( LDAP_DEBUG_TRACE, "operation initiated %d\n", rc, 0,
		    0 );

		if ( rc == OK )
			return( 0 );

		/* 
		 * the operation was not invoked - try rebinding, then 
		 * try it again.
		 */

		(void) dap_unbind( m->m_conn->c_ad );

		if ( do_bind_real( m->m_conn, &bound, &matched )
		    != LDAP_SUCCESS )
			break;
	}

	m->m_conn->c_ad = -1;

	return( LDAP_UNAVAILABLE );	/* DSA was unreachable */
}

#ifdef LDAP_DEBUG
int
trace_ber( tag, len, ber, trace_file, prepend )
    int   tag;
    int   len;
    char  *ber;
    FILE  *trace_file;
    int	  prepend;
{
	unsigned char   *buf;
	PS              input_ps  = NULLPS;
	PE              pe;
	int             result = -1;

	Debug( LDAP_DEBUG_BER, "trace_ber(tag=%#x, ber=%#lx, len=%d)\n", tag,
	    (unsigned long) ber, len );

	if ( (buf = (unsigned char *) malloc( len + 6 )) == NULL ) {
		fprintf( trace_file, "Unable to allocate memory\n" );
	} else {
		if ( prepend ) {
			buf[0] = tag;
			buf[1] = 0x84;
			buf[2] = len >> 24;
			buf[3] = len >> 16;
			buf[4] = len >> 8;
			buf[5] = len;
			memcpy( buf + 6, ber, len );
		} else {
			memcpy( buf, ber, len );
		}
		if ( (input_ps = ps_alloc( str_open )) == NULLPS )
			fprintf( trace_file, "ps_alloc failed\n" );
		else if ( str_setup( input_ps, buf, len + 6, 1 ) != OK )
			fprintf( trace_file, "str_setup\n" );
		else if ( (pe = ps2pe( input_ps )) == NULLPE )
			fprintf(trace_file, "ps2pe: %s\n",
			    ps_error( input_ps->ps_errno ) );
		else {
		      vsetfp( trace_file, NULL );
		      vunknown( pe );
		      pe_free( pe );
		      result = 0;
		}
		free( buf );
	}

      if ( input_ps )
              ps_free( input_ps );

      return( result );
}
#endif
