/*
 * Copyright (c) 1990 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 */
/*
 * Some code fragments to run from inetd stolen from the University
 * of Minnesota gopher distribution, which had this copyright on it:
 *
 * Part of the Internet Gopher program, copyright (C) 1991
 * University of Minnesota Microcomputer Workstation and Networks Center
 */

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/wait.h>
#include <signal.h>
#ifdef _AIX
#include <sys/select.h>
#endif
#ifdef __hpux
#include <syslog.h>
#define getdtablesize()	_NFILE
#else
#include <sys/syslog.h>
#endif
#include "lber.h"
#include "ldap.h"
#include "common.h"

#ifndef FD_SET
#define NFDBITS         32
#define FD_SETSIZE      32
#define FD_SET(n, p)    ((p)->fds_bits[(n)/NFDBITS] |= (1 << ((n) % NFDBITS)))
#define FD_CLR(n, p)    ((p)->fds_bits[(n)/NFDBITS] &= ~(1 << ((n) % NFDBITS)))
#define FD_ISSET(n, p)  ((p)->fds_bits[(n)/NFDBITS] & (1 << ((n) % NFDBITS)))
#define FD_ZERO(p)      bzero((char *)(p), sizeof(*(p)))
#endif

void log_and_exit();
static set_socket();
static wait4child();

#ifdef LDAP_DEBUG
int	ldap_debug;
#endif
int	version;
#ifdef OLDBROKEN
int	oldbroken;
#endif
int	dosyslog;
int	idletime = DEFAULT_TIMEOUT;
int	referral_connection_timeout = DEFAULT_REFERRAL_TIMEOUT;
struct timeval	conn_start_tv;
#ifdef KERBEROS
char	*krb_ldap_service = "ldapserver";
char	*krb_x500_service = "x500dsa";
char	*krb_x500_instance;
char	*krb_x500_nonce;
char	*kerberos_keyfile;
#endif

int	RunFromInetd = 0;

static usage( name )
char	*name;
{
	fprintf( stderr, "usage: %s [-d debuglvl] [-p port] [-l] [-c dsa] [-t timeout] [-r referraltimeout]", name );
	fprintf( stderr, " [-I]" );
#ifdef KERBEROS
	fprintf( stderr, " [-i dsainstance]" );
#endif
	fprintf( stderr, "\n" );
}

main (argc, argv)
int	argc;
char	**argv;
{
	int			s, ns;
	int			myport = LDAP_PORT;
	int			tblsize;
	int			i, pid;
	char			*myname;
	fd_set			readfds;
	struct hostent		*hp;
	struct sockaddr_in	from;
	int			fromlen;
	int			dsapargc;
	char			**dsapargv;
	char			title[80];
	int			wait4child();
	extern char		*optarg;
	extern char		**Argv;
	extern int		Argc;

#ifdef VMS
	/* Pick up socket from inetd-type server on VMS */
	if ( (ns = socket_from_server( NULL )) > 0 )
		RunFromInetd = 1;
#else
        /* Socket from inetd is usually 0 */
        ns = 0;
#endif

	/* for dsap_init */
        if ( (dsapargv = (char **) malloc( 4 * sizeof(char *) )) == NULL ) {
                perror( "malloc" );
                exit( 1 );
        }
        dsapargv[0] = argv[0];
        dsapargv[1] = 0;
        dsapargv[2] = 0;
        dsapargv[3] = 0;
        dsapargc = 1;
#ifdef KERBEROS
	kerberos_keyfile = "";
#endif

	/* process command line arguments */
	while ( (i = getopt( argc, argv, "d:lp:f:i:c:r:t:I" )) != EOF ) {
		switch ( i ) {
		case 'c':	/* specify dsa to contact */
			dsapargv[1] = "-call";
			dsapargv[2] = strdup( optarg );
			dsapargc = 3;
			break;

		case 'd':	/* turn on debugging */
#ifdef LDAP_DEBUG
			ldap_debug = atoi( optarg );
			if ( ldap_debug & LDAP_DEBUG_PACKETS )
				lber_debug = ldap_debug;
#else
			fprintf( stderr, "Not compiled with -DLDAP_DEBUG!\n" );
#endif
			break;

		case 'l':	/* do syslogging */
			dosyslog = 1;
			break;

		case 'p':	/* specify port number */
			myport = atoi( optarg );
			break;

		case 'r':	/* timeout for referral connections */
			referral_connection_timeout = atoi( optarg );
			break;

		case 't':	/* timeout for idle connections */
			idletime = atoi( optarg );
			break;

#ifdef KERBEROS
		case 'f':	/* kerberos key file */
			kerberos_keyfile = strdup( optarg );
			break;

		case 'i':	/* x500 dsa kerberos instance */
			if ( krb_x500_instance != NULL )
				free( krb_x500_instance );
			krb_x500_instance = strdup( optarg );
			break;
#endif

		case 'I':	/* Run from inetd */
			RunFromInetd = 1;
			break;

		default:
			usage( argv[0] );
			exit( 1 );
		}
	}

	/* 
	 * set up syslogging (if desired), detach from the terminal
	 * if stderr is redirected or no debugging is wanted, and
	 * then arrange to reap children that have exited
	 */

	if ( (myname = strrchr( argv[0], '/' )) == NULL )
		myname = strdup( argv[0] );
	else
		myname = strdup( myname + 1 );

	if ( dosyslog && openlog( myname, LOG_PID | LOG_NOWAIT, LOG_LOCAL4 )
	    < 0 ) {
		perror("openlog");
	}

	/* for setproctitle */
	Argv = argv;
	Argc = argc;
	if (!RunFromInetd) {
		setproctitle( "initializing" );
#ifndef VMS
		(void) detach();
#endif
		(void) signal( SIGCHLD, wait4child );
	}

	/* 
	 * load the syntax handlers, oidtables, and initialize some stuff,
	 * then start listening
	 */

	(void) quipu_syntaxes();
#ifdef LDAP_USE_PP
	(void) pp_quipu_init( argv[0] );
#endif
	(void) dsap_init( &dsapargc, &dsapargv );
	(void) get_syntaxes();
	if (RunFromInetd) {
		fromlen = sizeof(from);
		if ( getpeername( ns, (struct sockaddr *) &from, &fromlen )
		    == 0 ) {
			myport = ntohs( from.sin_port );

			hp = gethostbyaddr( (char *) &(from.sin_addr.s_addr),
			    sizeof(from.sin_addr.s_addr), AF_INET );
			Debug( LDAP_DEBUG_ARGS, "connection from %s (%s)\n",
			    (hp == NULL) ? "unknown" : hp->h_name,
			    inet_ntoa( from.sin_addr ), 0 );

			if ( dosyslog ) {
				syslog( LOG_INFO, "connection from %s (%s)",
				    (hp == NULL) ? "unknown" : hp->h_name,
				    inet_ntoa( from.sin_addr ) );
			}

			sprintf( title, "%s", hp == NULL ?
			    inet_ntoa( from.sin_addr ) : hp->h_name );
			setproctitle( title );
		}
		gettimeofday( &conn_start_tv, (struct timezone *) NULL );

		do_queries( ns );

		exit( 0 );
	}
	s = set_socket( myport );
	tblsize = getdtablesize();

	/*
	 * loop, wait for a connection, then fork off a child to handle it
	 */

	sprintf( title, "listening %d", myport );
	setproctitle( title );
	for ( ;; ) {
		FD_ZERO( &readfds );
		FD_SET( s, &readfds );

		if ( select( tblsize, &readfds, 0, 0, 0 ) < 1 ) {
#ifdef LDAP_DEBUG
			if ( ldap_debug ) perror( "select" );
#endif
			continue;
		}

		if ( ! FD_ISSET( s, &readfds ) ) {
			Debug( LDAP_DEBUG_ANY, "fd not ready\n", 0, 0, 0 );
			continue;
		}

		fromlen = sizeof(from);
		if ( (ns = accept( s, &from, &fromlen )) == -1 ) {
#ifdef LDAP_DEBUG
			if ( ldap_debug ) perror( "accept" );
#endif
			continue;
		}

		hp = gethostbyaddr( (char *) &(from.sin_addr.s_addr),
		    sizeof(from.sin_addr.s_addr), AF_INET );
		Debug( LDAP_DEBUG_ARGS, "connection from %s (%s)\n",
		    (hp == NULL) ? "unknown" : hp->h_name,
		    inet_ntoa( from.sin_addr ), 0 );

		if ( dosyslog ) {
			syslog( LOG_INFO, "connection from %s (%s)",
			    (hp == NULL) ? "unknown" : hp->h_name,
			    inet_ntoa( from.sin_addr ) );
		}

#ifdef VMS
		/* This is for debug on terminal on VMS */
		close( s );
		setproctitle( hp == NULL ? inet_ntoa( from.sin_addr ) :
		    hp->h_name );
		gettimeofday( &conn_start_tv, (struct timezone *) NULL );
		(void) signal( SIGPIPE, log_and_exit );

		do_queries( ns );
		/* NOT REACHED */
#endif

		switch( pid = fork() ) {
		case 0:         /* child */
			close( s );
			setproctitle( hp == NULL ? inet_ntoa( from.sin_addr ) :
			    hp->h_name );
			gettimeofday( &conn_start_tv, (struct timezone *) NULL );
			(void) signal( SIGPIPE, log_and_exit );

			do_queries( ns );
			break;

		case -1:        /* failed */
#ifdef LDAP_DEBUG
			if ( ldap_debug ) perror( "fork" );
#endif
			break;

		default:        /* parent */
			close( ns );
			Debug( LDAP_DEBUG_TRACE, "forked child %d\n", pid, 0,
			    0 );
			break;
		}
	}
	/* NOT REACHED */
}

do_queries( clientsock )
int	clientsock;
{
	int		tblsize;
	fd_set		readfds;
	int		rc, i;
	struct timeval	timeout;
	Sockbuf		sb;

	/*
	 * Loop, wait for a request from the client or a response from
	 * a dsa, then handle it.  Dsap_ad is always a connection to the
	 * "default" dsa.  Other connections can be made as a result of
	 * a referral being chased down.  These association descriptors
	 * are kept track of with the message that caused the referral.
	 * The set_dsa_fds() routine traverses the list of outstanding
	 * messages, setting the appropriate bits in readfds.
	 */

	conn_init();

	sb.sb_sd = clientsock;
	sb.sb_ber.ber_buf = NULL;
	sb.sb_ber.ber_ptr = NULL;
	sb.sb_ber.ber_end = NULL;

	tblsize = getdtablesize();
	timeout.tv_sec = idletime;
	timeout.tv_usec = 0;
	for ( ;; ) {
		struct conn		*dsaconn;
		extern struct conn	*conns;

		FD_ZERO( &readfds );
		FD_SET( clientsock, &readfds );
		conn_setfds( &readfds );

#ifdef LDAP_DEBUG
		if ( ldap_debug & LDAP_DEBUG_CONNS ) {
			Debug( LDAP_DEBUG_CONNS, "FDLIST:", 0, 0, 0 );
			for ( i = 0; i < tblsize; i++ ) {
				if ( FD_ISSET( i, &readfds ) ) {
					Debug( LDAP_DEBUG_CONNS, " %d", i, 0,
					    0);
				}
			}
			Debug( LDAP_DEBUG_CONNS, "\n", 0, 0, 0 );
		}
#endif

		/* 
		 * hack - because of lber buffering, there might be stuff
		 * already waiting for us on the client sock.
		 */

		if ( sb.sb_ber.ber_ptr >= sb.sb_ber.ber_end ) {
			if ( (rc = select( tblsize, &readfds, 0, 0, &timeout ))
			    < 1 ) {
#ifdef LDAP_DEBUG
				if ( ldap_debug ) perror( "select" );
#endif
				if ( rc == 0 )
					log_and_exit( 0 ); /* idle timeout */

				Debug( LDAP_DEBUG_ANY, "select returns %d!\n",
				    rc, 0, 0 );

				/* client gone away - we can too */
				if ( isclosed( clientsock ) )
					log_and_exit( 0 );

				/*
				 * check if a dsa conn has gone away -
				 * mark it bad if so
				 */
				conn_badfds();

				continue;
			}
		}

		if ( sb.sb_ber.ber_ptr < sb.sb_ber.ber_end ||
		    FD_ISSET( clientsock, &readfds ) ) {
			client_request( &sb, conns );
		} else {
			if ( (dsaconn = conn_getfd( &readfds )) == NULL ) {
				Debug( LDAP_DEBUG_ANY, "No DSA activity!\n",
				    0, 0, 0 );
				continue;
			}

			dsa_response( dsaconn, &sb );
		}
	}
	/* NOT REACHED */
}

static set_socket( port )
int	port;
{
	int			s, i;
	struct sockaddr_in	addr;

	if ( (s = socket( AF_INET, SOCK_STREAM, 0 )) == -1 ) {
                perror( "socket" );
                exit( 1 );
        }

        /* set option so clients can't keep us from coming back up */
	i = 1;
        if ( setsockopt( s, SOL_SOCKET, SO_REUSEADDR, &i, sizeof(i) ) < 0 ) {
                perror( "setsockopt" );
                exit( 1 );
        }

        /* bind to a name */
        addr.sin_family = AF_INET;
        addr.sin_addr.s_addr = INADDR_ANY;
        addr.sin_port = htons( port );
        if ( bind( s, &addr, sizeof(addr) ) ) {
                perror( "bind" );
                exit( 1 );
        }

	/* listen for connections */
        if ( listen( s, 5 ) == -1 ) {
                perror( "listen" );
                exit( 1 );
        }
 
	Debug( LDAP_DEBUG_TRACE, "listening on port %d\n", port, 0, 0 );

	return( s );
}

static wait4child()
{
#if defined(SunOS) && SunOS < 40
        union wait status;
#else
        int     status;
#endif

	Debug( LDAP_DEBUG_TRACE, "parent: catching child status\n", 0, 0, 0 );

        while ( wait3( &status, WNOHANG | WUNTRACED, 0 ) > 0 )
                ;       /* NULL */

#ifdef __hpux
	(void) signal( SIGCHLD, wait4child );
#endif
}


void
log_and_exit( exitcode )
	int	exitcode;
{
	struct timeval	tv;

	if ( dosyslog ) {
		gettimeofday( &tv, (struct timezone *)NULL );
		syslog( LOG_INFO, "TCP closed %d seconds,  exit(%d)",
		    tv.tv_sec - conn_start_tv.tv_sec, exitcode );
	}

	exit( exitcode );
}
