/*
 * Copyright (c) 1990 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <quipu/commonarg.h>
#include <quipu/attrvalue.h>
#include <quipu/ds_error.h>
#include <quipu/bind.h>
#include <quipu/compare.h>
#include "lber.h"
#include "ldap.h"
#include "common.h"

#ifdef OLDBROKEN
extern int 	oldbroken;
#define BINDTAG	(oldbroken ? OLD_LDAP_RES_BIND : LDAP_RES_BIND)
#else
#define BINDTAG	LDAP_RES_BIND
#endif

/*
 * do_bind - perform an X.500 bind operation.  Since we always respond
 * to the request in here, always return 0 to signify the incoming message
 * can be discarded.
 */

do_bind( clientsb, m, ber, bound )
    Sockbuf	*clientsb;
    struct msg	*m;
    BerElement	*ber;
    int		*bound;
{
	int		err, method;
	long		len;
	int		msgid = m->m_msgid;
	char		*dn, *pw;
	char		*matched;
	struct PSAPaddr	*addr, *psap_cpy();
	extern char	*dsa_address;
	extern int	version;

	Debug( LDAP_DEBUG_TRACE, "do_bind\n", 0, 0, 0 );

	/*
	 * Parse the bind request.  It looks like this:
	 *	BindRequest ::= SEQUENCE {
	 *		version		INTEGER,		 -- version
	 *		name		DistinguishedName,	 -- dn
	 *		authentication	CHOICE {
	 *			simple		[0] OCTET STRING -- passwd
	 *			krbv42ldap	[1] OCTET STRING
	 *			krbv42dsa	[1] OCTET STRING
	 *		}
	 *	}
	 */

	if ( ber_scanf( ber, "{ia", &version, &dn ) == -1 ) {
		Debug( LDAP_DEBUG_ANY, "ber_scanf failed\n", 0, 0, 0 );
		send_ldap_result( clientsb, BINDTAG, msgid,
		    LDAP_PROTOCOL_ERROR, "Decoding error" );
		return( 0 );
	}
#ifdef OLDBROKEN
	if ( oldbroken )
		method = ber_peek_tag( ber, &len );
	else
#endif
		method = ber_skip_tag( ber, &len );
	if ( ber_scanf( ber, "la}", &len, &pw ) == -1 ) {
		Debug( LDAP_DEBUG_ANY, "ber_scanf2 failed\n", 0, 0, 0 );
		send_ldap_result( clientsb, BINDTAG, msgid,
		    LDAP_PROTOCOL_ERROR, "Decoding error" );
		return( 0 );
	}

	if ( version != LDAP_VERSION1 && version != LDAP_VERSION2 ) {
		Debug( LDAP_DEBUG_ANY, "unknown version %d\n", version, 0, 0 );
		send_ldap_result( clientsb, BINDTAG, msgid,
		    LDAP_PROTOCOL_ERROR, "Version not supported" );
		return( 0 );
	}

	Debug( LDAP_DEBUG_ARGS, "do_bind: version %d dn (%s) method %d\n",
	    version, dn, method );

	if ( dsa_address == NULL || (addr = str2paddr( dsa_address ))
	    == NULLPA ) {
		char	buf[256];

		sprintf( buf, "Bad DSA address (%s)", dsa_address ?
		    dsa_address : "NULL" );
		send_ldap_result( clientsb, BINDTAG, msgid,
		    LDAP_OPERATIONS_ERROR, buf );
		return( 0 );
	}

	if ( m->m_conn->c_dn )
		free( m->m_conn->c_dn );
	if ( m->m_conn->c_cred )
		free( m->m_conn->c_cred );
	if ( m->m_conn->c_paddr )
		free( (char *) m->m_conn->c_paddr );
	m->m_conn->c_dn = dn;
	m->m_conn->c_cred = pw;
	m->m_conn->c_credlen = len;
	m->m_conn->c_method = method;
	m->m_conn->c_paddr = psap_cpy( addr );

	err = do_bind_real( m->m_conn, bound, &matched );

	send_ldap_result_2( clientsb, BINDTAG, msgid, err, matched, "" );

	if ( matched != NULL )
		free( matched );

	return( 0 );
}

do_bind_real( dsaconn, bound, matched )
    struct conn	*dsaconn;
    int		*bound;
    char	**matched;
{
	struct ds_bind_arg	ba;
	struct ds_bind_arg	br;
	struct ds_bind_error	be;
	struct DSError		dse;
	char			*dn = dsaconn->c_dn;
	char			*pw = dsaconn->c_cred;
	int			err;
#ifdef KERBEROS
	u_long			nonce;
#endif
	extern DN		ldap_str2dn();

	Debug( LDAP_DEBUG_TRACE, "do_bind_real\n", 0, 0, 0 );

	*matched = NULL;
	if ( (ba.dba_dn = ldap_str2dn( dn )) == NULLDN && *dn != '\0' ) {
		Debug( LDAP_DEBUG_ANY, "ldap_str2dn (%s) failed\n", dn, 0, 0 );
		return( LDAP_INVALID_DN_SYNTAX );
	}

	switch ( dsaconn->c_method ) {
#ifdef OLDBROKEN
	case OLD_LDAP_AUTH_SIMPLE:
#endif
	case LDAP_AUTH_SIMPLE:	/* x.500 simple authentication */
		ba.dba_passwd_len = strlen( pw );
		strcpy( ba.dba_passwd, pw );
		ba.dba_auth_type = (ba.dba_passwd_len == 0 ? DBA_AUTH_NONE
		    : DBA_AUTH_SIMPLE);
		ba.dba_version = DBA_VERSION_V1988;
		break;

#ifdef KERBEROS
#ifdef OLDBROKEN
	case OLD_LDAP_AUTH_KRBV4:
#endif
	case LDAP_AUTH_KRBV41:	/* kerberos authentication to ldap server */
		return( kerberosv4_ldap_auth( dsaconn->c_cred,
		    dsaconn->c_credlen ) );
		break;

#ifdef OLDBROKEN
	case OLD_LDAP_AUTH_KRBV42:
#endif
	case LDAP_AUTH_KRBV42:	/* kerberos authentication to x500 dsa */
		if ( (err = kerberosv4_bindarg( &ba, ba.dba_dn, dsaconn->c_cred,
		    dsaconn->c_credlen, &nonce )) != 0 )
			return( err );
		break;
#endif

	default:
		return( LDAP_PROTOCOL_ERROR );
		break;
	}

	if ( dsaconn->c_ad != -1 )
		dap_unbind( dsaconn->c_ad );

	Debug( LDAP_DEBUG_TRACE, "dap_bind to dsa (%s)...\n", paddr2str(
	    dsaconn->c_paddr, NULLNA ), 0, 0 );

	if ( dap_bind( &dsaconn->c_ad, &ba, &be, &br, dsaconn->c_paddr )
	    != DS_OK ) {
		if ( ba.dba_dn != NULLDN )
			dn_free( ba.dba_dn );

		if ( be.dbe_type == DBE_TYPE_SERVICE ) {
			dse.dse_type = DSE_SERVICEERROR;
			dse.ERR_SERVICE.DSE_sv_problem = be.dbe_value;
		} else if ( be.dbe_type == DBE_TYPE_SECURITY ) {
			dse.dse_type = DSE_SECURITYERROR;
			dse.ERR_SECURITY.DSE_sc_problem = be.dbe_value;
		} else {
			dse.dse_type = DSE_REMOTEERROR;
		}
		err = x500err2ldaperr( &dse, matched );

#ifdef LDAP_DEBUG
		if ( ldap_debug )
			print_error( &dse );	/* prints and then frees */
		else
#endif
			ds_error_free( &dse );

		dsaconn->c_ad = -1;

		return( err );
	}
	if ( ba.dba_dn != NULLDN )
		dn_free( ba.dba_dn );

	Debug( LDAP_DEBUG_TRACE, "dap_bind successful\n", 0, 0, 0 );

#ifdef KERBEROS
/* XXX why doesn't this work??
	if ( dsaconn->c_method == LDAP_AUTH_KRBV42 &&
	    kerberos_check_mutual( &br, nonce ) != 0 ) {
		Debug( LDAP_DEBUG_ANY, "Mutual authentication failed\n", 0, 0,
		    0 );
		return( LDAP_INVALID_CREDENTIALS );
	}
*/
#endif

	*bound = 1;

	return( LDAP_SUCCESS );
}
