/*
 * Copyright (c) 1990 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 */

#include "lber.h"
#include "ldap.h"
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/syslog.h>
#include <sys/wait.h>
#include <signal.h>

/*
 ***************************************************************************
 * If you are not a U-M site, you probably want to tailor the following:
 ***************************************************************************
 */

/* the x500 DN that the finger daemon binds to - can be NULL */
#define DAPUSER			"cn=Finger, ou=Miscellaneous Servers, o=University of Michigan, c=US"

/* subtree to search */
#define DEFAULT_BASE		"o=University of Michigan, c=US"

/* where the filter configuration file lives */
#define DEFAULT_FILTERCONF	"/usr/local/etc/ldapfilter.conf"

/* for various error messages */
#define REPORT_ERRORS_TO	"x500@umich.edu"

/* message to print when we don't find anybody */
#define NOMATCH_MESSAGE		"Search failed to find anything.  Currently, this service contains information\r\n\
only on University of Michigan faculty, staff and students.  Other likely\r\n\
places to finger are:\r\n\
    um.cc.umich.edu\r\n\
    azure.engin.umich.edu\r\n\
    med.umich.edu\r\n"

/* message to print in lieu of an email address */
#define NOEMAIL_MESSAGE		"                    None registered in this service.  Try fingering at:\r\n                    um.cc.umich.edu, azure.engin.umich.edu, med.umich.edu.\r\n"

/*
 ***************************************************************************
 * The rest of this stuff does probably not need to be changed
 ***************************************************************************
 */

#define DEFAULT_LDAPHOST	"localhost"
#define DEFAULT_PORT		79
#define DEFAULT_SIZELIMIT	50

/* what to exec when we get a null request */
#define FINGER			"/usr/ucb/finger"

#ifndef FD_SET
#define NFDBITS         32
#define FD_SETSIZE      32
#define FD_SET(n, p)    ((p)->fds_bits[(n)/NFDBITS] |= (1 << ((n) % NFDBITS)))
#define FD_CLR(n, p)    ((p)->fds_bits[(n)/NFDBITS] &= ~(1 << ((n) % NFDBITS)))
#define FD_ISSET(n, p)  ((p)->fds_bits[(n)/NFDBITS] & (1 << ((n) % NFDBITS)))
#define FD_ZERO(p)      bzero((char *)(p), sizeof(*(p)))
#endif

int	debug;
int	dosyslog = 1;

char	*ldaphost = DEFAULT_LDAPHOST;
char	*base = DEFAULT_BASE;
int	strip = 3;
int	searchaliases = 1;
int	deref;
int	sizelimit;
LDAPFiltDesc	*filtd;

static do_query();
static do_search();
static do_read();
static print_attr();

static usage( name )
char	*name;
{
	fprintf( stderr, "usage: %s [-d debuglevel] [-p port] [-l] [-x ldaphost] [-b searchbase] [-a] [-s stripcount] [-z sizelimit] [-f filterfile]\r\n" );
	exit( 1 );
}

main (argc, argv)
int	argc;
char	**argv;
{
	int			s, ns, rc;
	int			port = -1;
	int			tblsize;
	int			i, pid;
	char			*myname, *filterfile = DEFAULT_FILTERCONF;
	unsigned long		mypeer = -1;
	fd_set			readfds;
	struct hostent		*hp;
	struct timeval		timeout;
	struct sockaddr_in	peername;
	int			peernamelen;
	int			wait4child();
	extern char		*optarg;

	deref = LDAP_DEREF_ALWAYS;
	while ( (i = getopt( argc, argv, "ab:d:f:lp:s:x:" )) != EOF ) {
		switch( i ) {
		case 'a':	/* do not deref aliases when searching */
			deref = LDAP_DEREF_FINDING;
			break;

		case 'b':	/* search base */
			base = strdup( optarg );
			break;

		case 'd':	/* turn on debugging */
			debug = atoi( optarg );
			break;

		case 'f':	/* ldap filter file */
			filterfile = strdup( optarg );
			break;

		case 'l':	/* do syslogging */
			dosyslog = 1;
			break;

		case 's':	/* number of components to strip off dns */
			strip = atoi( optarg );
			break;

		case 'x':	/* specify ldap host */
			ldaphost = strdup( optarg );
			break;

		case 'z':	/* size limit */
			sizelimit = atoi( optarg );
			break;

		default:
			usage( argv[0] );
		}
	}

	peernamelen = sizeof(peername);
	if ( getpeername( 0, &peername, &peernamelen ) != 0 ) {
		perror( "getpeername" );
		exit( 1 );
	}
	mypeer = (unsigned long) peername.sin_addr.s_addr;

	printf( "X.500 Finger Service...\r\n" );
	fflush( stdout );

	if ( (myname = strrchr( argv[0], '/' )) == NULL )
		myname = strdup( argv[0] );
	else
		myname = strdup( myname + 1 );

	if ( dosyslog && openlog( myname, LOG_PID | LOG_NOWAIT, LOG_LOCAL4 )
	    < 0 ) {
		perror("openlog");
	}

	if ( dosyslog && mypeer != -1 ) {
		struct in_addr	addr;

		hp = gethostbyaddr( (char *) &mypeer, sizeof(mypeer), AF_INET );
		addr.s_addr = mypeer;
		syslog( LOG_INFO, "connection from %s (%s)", (hp == NULL) ?
		    "unknown" : hp->h_name, inet_ntoa( addr ) );
	}

	if ( (filtd = ldap_init_getfilter( filterfile )) == NULL ) {
		fprintf( stderr, "Cannot open filter file (%s)\n", filterfile );
		exit( 1 );
	}

	do_query();
}

static do_query()
{
	char		buf[256];
	int		len, i;
	int		rc, tblsize;
	struct timeval	timeout;
	fd_set		readfds;
	LDAP		*ld;

	if ( (ld = ldap_open( ldaphost, LDAP_PORT )) == NULL ) {
		perror( "ldap_open" );
		exit( 1 );
	}
	ld->ld_sizelimit = sizelimit ? sizelimit : DEFAULT_SIZELIMIT;
	ld->ld_deref = deref;

	if ( ldap_simple_bind_s( ld, DAPUSER, NULL ) != LDAP_SUCCESS ) {
		fprintf( stderr, "Sorry, the X.500 service is temporarily unavailable.\n" );
		fprintf( stderr, "Please try again later.\n" );
		ldap_perror( ld, "ldap_simple_bind_s" );
		exit( 1 );
	}

	tblsize = getdtablesize();
	timeout.tv_sec = 60;
	timeout.tv_usec = 0;
	FD_ZERO( &readfds );
	FD_SET( fileno( stdin ), &readfds );

	if ( (rc = select( tblsize, &readfds, 0, 0, &timeout )) <= 0 ) {
		if ( rc < 0 )
			perror( "select" );
		else
			fprintf( stderr, "connection timed out on input\r\n" );
		exit( 1 );
	}

	if ( fgets( buf, sizeof(buf), stdin ) == NULL )
		exit( 1 );

	len = strlen( buf );
	if ( debug ) {
		fprintf( stderr, "got %d bytes\r\n", len );
#ifdef LDAP_DEBUG
		lber_bprint( buf, len );
#endif
	}

	/* strip off \r \n */
	if ( buf[len - 1] == '\n' ) {
		buf[len - 1] = '\0';
		len--;
	}
	if ( buf[len - 1] == '\r' ) {
		buf[len - 1] = '\0';
		len--;
	}

	if ( len == 0 ) {
		printf( "No campus-wide login information available.  Info for this machine only:\r\n" );
		fflush( stdout );
		execl( FINGER, FINGER, NULL );
	} else {
		char	*p;

		/* skip and ignore stinking /w */
		if ( strncmp( buf, "/W ", 2 ) == 0 ) {
			p = buf + 2;
		} else {
			p = buf;
		}

		for ( ; *p && isspace( *p ); p++ )
			;	/* NULL */

		do_search( ld, p );
	}
}

static do_search( ld, buf )
LDAP	*ld;
char	*buf;
{
	char	*dn, *rdn;
	char	**title;
	int	rc, matches, i;
	struct timeval	tv;
	LDAPFiltInfo	*fi;
	LDAPMessage	*result, *e;
	static char	*attrs[] = { "cn", "title", 0 };

	for ( fi = ldap_getfirstfilter( filtd, "finger", buf ); fi != NULL;
	    fi = ldap_getnextfilter( filtd ) ) {
		if ( (rc = ldap_search_s( ld, base, LDAP_SCOPE_SUBTREE,
		    fi->lfi_filter, attrs, 0, &result )) != LDAP_SUCCESS
		    && rc != LDAP_SIZELIMIT_EXCEEDED
		    && rc != LDAP_TIMELIMIT_EXCEEDED
		) {
			fprintf( stderr, "Sorry, an error occured during the search.\n" );
			fprintf( stderr, "The X.500 service may be temporarily unavailable.\n" );
			fprintf( stderr, "Please try again later.\n" );
			ldap_perror( ld, "ldap_search_s" );
			exit( 1 );
		}

		if ( (matches = ldap_count_entries( ld, result )) != 0 )
			break;
	}

	if ( rc == LDAP_SIZELIMIT_EXCEEDED )
		printf( "(Partial results - a size limit was exceeded)\r\n" );
	else if ( rc == LDAP_TIMELIMIT_EXCEEDED )
		printf( "(Partial results - a time limit was exceeded)\r\n" );

	if ( matches == 1 ) {
		printf( "One %s match found for \"%s\":\r\n", fi->lfi_desc,
		    buf );
		fflush( stdout );

		e = ldap_first_entry( ld, result );
		do_read( ld, ldap_get_dn( ld, e ) );
	} else if ( matches > 1 ) {
		printf( "%d %s matches for \"%s\":\r\n", matches,
		    fi->lfi_desc, buf );
		fflush( stdout );

		for ( e = ldap_first_entry( ld, result ); e != NULL;
		    e = ldap_next_entry( ld, e ) ) {
			char	*p;

			dn = ldap_get_dn( ld, e );
			rdn = dn;
			if ( (p = strchr( dn, ',' )) != NULL )
				*p = '\0';
			while ( *rdn && *rdn != '=' )
				rdn++;
			if ( *rdn )
				rdn++;

			/* hack attack */
			for ( i = 0; buf[i] != '\0'; i++ ) {
				if ( buf[i] == '.' || buf[i] == '_' )
					buf[i] = ' ';
			}
			if ( strcasecmp( rdn, buf ) == 0 ) {
				char	**cn;
				char	*s;
				int	i, last;

				cn = ldap_get_values( ld, e, "cn" );
				for ( i = 0; cn[i] != NULL; i++ ) {
					last = strlen( cn[i] ) - 1;
					if ( isdigit( cn[i][last] ) ) {
						rdn = strdup( cn[i] );
						break;
					}
				}
			}
					
			title = ldap_get_values( ld, e, "title" );

			printf( "  %-20s    %s\r\n", rdn,
			    title ? title[0] : "" );
			if ( title != NULL ) {
				for ( i = 1; title[i] != NULL; i++ )
					printf( "  %-20s    %s\r\n", "",
					    title[i] );
			}
			fflush( stdout );

			if ( title != NULL )
				ldap_value_free( title );

			free( dn );
		}
	} else if ( matches == 0 ) {
		printf( NOMATCH_MESSAGE );
		fflush( stdout );
	} else {
		fprintf( stderr, "error return from ldap_count_entries\r\n" );
		exit( 1 );
	}

	ldap_unbind( ld );
}

static print_attr( ld, fp, label, attr, e, multiline, ufnfmt, isgroup )
LDAP		*ld;
FILE		*fp;
char		*label;
char		*attr;
LDAPMessage	*e;
int		multiline;
int		ufnfmt;
int		isgroup;
{
	char	**val;
	int	i, gotone = 0;

	if ( (val = ldap_get_values( ld, e, attr )) == NULL ) {
		if ( !isgroup && strcmp( attr, "mail" ) == 0 ) {
			fprintf( fp, "%-19s\r\n", label );
			fprintf( fp, NOEMAIL_MESSAGE );
		}
		return;
	}

	fprintf( fp, "%-19s\r\n", label );
	for ( i = 0; val[i] != NULL; i++ ) {
		if ( multiline ) {
			char	*s, *p;

			if ( gotone )
				fprintf( fp, "%-19s\r\n", label );
			p = s = val[i];
			while ( s = strchr( s, '$' ) ) {
				*s++ = '\0';
				while ( isspace( *s ) )
					s++;
				fprintf( fp, "                    %s\r\n", p );
				p = s;
			}
			fprintf( fp, "                    %s\r\n", p );
			gotone = 1;
		} else {
			fprintf( fp, "                    %s\r\n", ufnfmt ?
			    ldap_dn2ufn( val[i] ) : val[i] );
		}
	}
	ldap_value_free( val );
}

static do_read( ld, dn )
LDAP	*ld;
char	*dn;
{
	char		*ufn;
	int		rc, i, isgroup;
	char		*s;
	char		**val;
	struct timeval	tv;
	LDAPMessage	*result, *e;
	static char	*attrs[] = { "cn", "postalAddress", "uid", "title",
				     "mail", "telephoneNumber", "pager",
				     "facsimileTelephoneNumber", "objectClass",
				     "member", "owner", "joinable",
#ifdef UOFM
				     "multiLineDescription",
#else
				     "description",
#endif
				     0 };

	if ( (rc = ldap_search_s( ld, dn, LDAP_SCOPE_BASE, "objectclass=*",
	    attrs, 0, &result )) != LDAP_SUCCESS
	    && rc != LDAP_SIZELIMIT_EXCEEDED
	    && rc != LDAP_TIMELIMIT_EXCEEDED )
	{
		fprintf( stderr, "Sorry, an error occured reading results.\n" );
		fprintf( stderr, "The X.500 service may be temporarily unavailable.\n" );
		fprintf( stderr, "Please try again later.\n" );
		ldap_perror( ld, "ldap_search_s" );
		exit( 1 );
	}

	if ( ldap_count_entries( ld, result ) != 1 ) {
		printf( "Sorry, an error occurred parsing the entry.  Please report this to %s\n", REPORT_ERRORS_TO );
		exit( 1 );
	}

	e = ldap_first_entry( ld, result );
	dn = ldap_get_dn( ld, e );
	ufn = ldap_dn2ufn( dn );
	for ( i = 0; i < strip; i++ ) {
		if ( (s = strrchr( ufn, ',' )) == NULL )
			break;
		*s = '\0';
	}

	if ( (val = ldap_get_values( ld, e, "objectClass" )) == NULL ) {
		ldap_perror( ld, "ldap_get_values objectClass" );
		exit( 1 );
	}
	isgroup = 0;
	for ( i = 0; val[i] != NULL; i++ ) {
		if ( strcasecmp( val[i], "rfc822MailGroup" ) == 0 ) {
			isgroup = 1;
			break;
		}
	}
	ldap_value_free( val );

	fprintf( stdout, "\"%s\"\r\n", ufn );
	print_attr( ld, stdout, "  Also known as:", "cn", e, 0, 0, isgroup );
	if ( !isgroup )
		print_attr( ld, stdout, "  E-mail address:", "mail", e, 0, 0,
		    isgroup );
	print_attr( ld, stdout, "  Fax number:", "facsimileTelephoneNumber", e,
	    0, 0, isgroup );
	print_attr( ld, stdout, "  Pager number:", "pager", e, 0, 0, isgroup );
	print_attr( ld, stdout, "  Business phone:", "telephoneNumber", e, 0,
	    0, isgroup );
	print_attr( ld, stdout, "  Business address:", "postalAddress", e, 1,
	    0, isgroup );
	print_attr( ld, stdout, "  Title:", "title", e, 0, 0, isgroup );
	print_attr( ld, stdout, "  Uniqname:", "uid", e, 0, 0, isgroup );
#ifdef UOFM
	print_attr( ld, stdout, "  Description:", "multiLineDescription", e, 1,
	    0, isgroup );
#else
	print_attr( ld, stdout, "  Description:", "description", e, 1, 0,
	    isgroup );
#endif
	if ( isgroup ) {
		print_attr( ld, stdout, "  rfc822 Members:", "mail", e, 0, 0,
		    isgroup );
		print_attr( ld, stdout, "  X.500 Members:", "member", e, 0, 1,
		    isgroup );
		print_attr( ld, stdout, "  Others may join:", "joinable", e,
		    0, 0, isgroup );
		print_attr( ld, stdout, "  Owner:", "owner", e, 0, 1, isgroup );
	}

	free( ufn );
}
